package com.xebialabs.xlrelease.support.report

import com.xebialabs.xlplatform.support.report.ReportDataProvider
import com.xebialabs.xlrelease.config.ArchivingSettingsManager
import com.xebialabs.xlrelease.domain.status.{ReleaseStatus, TaskStatus}
import com.xebialabs.xlrelease.support.report.repository.DataStatisticsRepository
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Component

import java.util
import scala.beans.BeanProperty
import scala.collection.mutable
import scala.jdk.CollectionConverters._
import scala.math.BigDecimal.RoundingMode

@Component
class DataStatisticsProvider @Autowired()(val repository: DataStatisticsRepository,
                                          val archivingSettingsManager: ArchivingSettingsManager) extends ReportDataProvider {

  val name: String = "data"

  //noinspection ScalaStyle
  override def collectStatistics: Map[String, Any] = {
    Map[String, Any]("data" -> mutable.Map[String, Any](
      "releases" -> collectReleaseUsageStatistics,
      "tasks" -> collectTaskUsageStatistics,
      "templates" -> collectTemplateUsageStatistics,
      "triggers" -> collectTriggerUsageStatistics,
      "folders" -> collectFolderUsageStatistics,
      "variables" -> collectVariableUsageStatistics,
      "users" -> collectUserUsageStatistics,
      "dashboards" -> collectDashboardUsageStatistics,
      "groups" -> collectGroupUsageStatistics,
      "deliveries" -> collectDeliveryUsageStatistics,
      "configurations" -> collectConfigurationUsageStatistics,
      "applications" -> collectApplicationUsageStatistics,
      "environments" -> collectEnvironmentsUsageStatistics,
      "calendarEntries" -> collectCalendarUsageStatistics,
      "archive" -> collectArchiveUsageStatistics,
      "report" -> collectReportUsageStatistics
    ).asJava)
  }

  def collectReleaseUsageStatistics: ReleaseUsageStatistics = {
    val statistics = new ReleaseUsageStatistics
    statistics.plannedReleasesTotal = repository.countReleasesByStatus(ReleaseStatus.PLANNED.value())
    statistics.inProgressReleasesTotal = repository.countReleasesByStatus(ReleaseStatus.IN_PROGRESS.value())
    statistics.pausedReleasesTotal = repository.countReleasesByStatus(ReleaseStatus.PAUSED.value())
    statistics.failingReleasesTotal = repository.countReleasesByStatus(ReleaseStatus.FAILING.value())
    statistics.failedReleasesTotal = repository.countReleasesByStatus(ReleaseStatus.FAILED.value())
    statistics.completedReleasesTotal = repository.countReleasesByStatus(ReleaseStatus.COMPLETED.value())
    statistics.abortedReleasesTotal = repository.countReleasesByStatus(ReleaseStatus.ABORTED.value())
    statistics.releases24HoursTotal = repository.countReleasesInLastDays(1)
    statistics.releases30DaysTotal = repository.countReleasesInLastDays(30)
    statistics.completionRateLastMonth = repository.calculateCompletionRateLastMonth()
    statistics.averageReleasesPerDayLastYear = repository.averageReleasesInLastDays(365)
    statistics.averageReleasesPerMonthLastYear = repository.averageReleasesPerMonthInLastYear()
    statistics.releasesWithDependencies = repository.countReleasesWithDependencies()
    statistics.dependenciesTotal = repository.countDependencies()
    statistics.averageDependenciesPerReleases = repository.averageDependenciesPerRelease()
    statistics.maxDependenciesInRelease = repository.findMaxDependenciesInRelease()
    statistics.releaseTagsTotal = repository.countDistinctReleaseTags()
    statistics
  }

  def collectTaskUsageStatistics: TaskUsageStatistics = {
    val statistics = new TaskUsageStatistics
    statistics.plannedTasksTotal = repository.countPlannedTasks()
    statistics.pendingTasksTotal = repository.countTasksByStatuses(Seq(TaskStatus.PENDING))
    statistics.inProgressTasksTotal = repository.countTasksByStatuses(Seq(
      TaskStatus.IN_PROGRESS, TaskStatus.WAITING_FOR_INPUT,
      TaskStatus.PRECONDITION_IN_PROGRESS, TaskStatus.FAILURE_HANDLER_IN_PROGRESS,
      TaskStatus.FACET_CHECK_IN_PROGRESS, TaskStatus.ABORT_SCRIPT_IN_PROGRESS)
    )
    statistics.failedTasksTotal = repository.countTasksByStatuses(Seq(TaskStatus.FAILED, TaskStatus.FAILING))
    statistics.skippedTasksTotal = repository.countTasksByStatuses(Seq(TaskStatus.SKIPPED, TaskStatus.SKIPPED_IN_ADVANCE))
    statistics.completedTasksTotal = repository.countTasksByStatuses(Seq(TaskStatus.COMPLETED, TaskStatus.COMPLETED_IN_ADVANCE))
    statistics.abortedTasksTotal = repository.countTasksByStatuses(Seq(TaskStatus.ABORTED))
    statistics.templateTasksTotal = repository.countTasksByStatuses(Seq(TaskStatus.PLANNED)) - statistics.plannedTasksTotal
    statistics.maxTasksInRelease = repository.findMaxTasksInRelease()
    statistics.averageTasksPerRelease = repository.averageTasksPerRelease()
    statistics.taskTypeOccurrences = repository.countTopTaskTypeOccurrences(25).asJava
    statistics.automationPercentage = calculateAutomationPercentage(repository.countAutomatedTasks(),repository.countTasks())
    statistics.taskTagsTotal = repository.countDistinctTasksTags()
    statistics.taskBackupsTotal = repository.countTaskBackups()
    statistics.commentsTotal = repository.countTaskComments()
    statistics.attachmentsTotal = repository.countAttachments()
    statistics
  }

  def collectUserUsageStatistics: UserUsageStatistics = {
    val statistics = new UserUsageStatistics
    statistics.internalUsersTotal = repository.countUsers()
    statistics.activeUsersLastDay = repository.countActiveUsersInLastDays(1)
    statistics.activeUsersLast30Days = repository.countActiveUsersInLastDays(30)
    statistics.activeUsersLast90Days = repository.countActiveUsersInLastDays(90)
    statistics.globalRolesTotal = repository.countGlobalRoles()
    statistics.usersTotal = repository.countUserProfiles()
    statistics.externalUsersTotal = statistics.usersTotal - statistics.internalUsersTotal
    statistics.enabledUsersTotal = repository.countEnabledUsers()
    statistics.teamsTotal = repository.countTeams()
    statistics.roleRolesTotal = repository.countRoleRoles()
    statistics.rolePrincipalsTotal = repository.countRolePrincipals()
    statistics.rolePermissionsTotal = repository.countRolePermissions()
    statistics
  }

  def collectTemplateUsageStatistics: TemplateUsageStatistics = {
    val statistics = new TemplateUsageStatistics
    statistics.templatesTotal = repository.countTemplates()
    statistics.templateRevisionsTotal = repository.countTemplateRevisions()
    statistics
  }

  def collectTriggerUsageStatistics: TriggerUsageStatistics = {
    val statistics = new TriggerUsageStatistics
    statistics.triggersTotal = repository.countTriggers()
    statistics.activeTriggersTotal = repository.countActiveTriggers()
    statistics
  }

  def collectFolderUsageStatistics: FolderUsageStatistics = {
    val statistics = new FolderUsageStatistics
    statistics.foldersTotal = repository.countFolders()
    statistics.maxDepthOfFolder = repository.maxFolderDepth()
    statistics
  }

  def collectVariableUsageStatistics: VariableUsageStatistics = {
    val statistics = new VariableUsageStatistics
    val folderVariableCount = repository.countFolderVariables()
    statistics.globalVariablesTotal = repository.countGlobalVariables()
    statistics.folderVariablesTotal = folderVariableCount
    statistics.averageVariablesPerFolder = avg(folderVariableCount, repository.countFolders())
    statistics
  }

  def collectDashboardUsageStatistics: DashboardUsageStatistics = {
    val statistics = new DashboardUsageStatistics
    statistics.globalDashboardsTotal = repository.countGlobalDashboards()
    statistics.folderDashboardsTotal = repository.countFolderDashboards()
    statistics
  }

  def collectGroupUsageStatistics: ReleaseGroupUsageStatistics = {
    val statistics = new ReleaseGroupUsageStatistics
    statistics.groupsTotal = repository.countReleaseGroups()
    statistics
  }

  def collectDeliveryUsageStatistics: DeliveryUsageStatistics = {
    val statistics = new DeliveryUsageStatistics
    statistics.patternsTotal = repository.countDeliveryPatterns()
    statistics.deliveriesTotal = repository.countDeliveries()
    statistics
  }

  def collectArchiveUsageStatistics: ArchiveUsageStatistics = {
    val statistics = new ArchiveUsageStatistics
    statistics.archivingEnabled = archivingSettingsManager.getEnabled
    statistics.archivingAge = archivingSettingsManager.getReleaseAgeToDeleteFromJcr / 24
    statistics.archivedReleasesTotal = repository.countArchivedReleases()
    statistics.prearchivedReleasesTotal = repository.countPreArchivedReleases()
    statistics.oldestArchivedRelease = Option(repository.oldestArchivedRelease()).getOrElse("None")
    statistics.archivedCompletedReleasesTotal = repository.countArchivedReleasesByState(ReleaseStatus.COMPLETED.value())
    statistics.archivedAbortedReleasesTotal = repository.countArchivedReleasesByState(ReleaseStatus.ABORTED.value())
    statistics.archivedPhasesTotal = repository.countArchivedPhases()
    statistics.archivedTasksTotal = repository.countArchivedTasks()
    statistics.archivedAttachmentsTotal = repository.countArchivedAttachments()
    statistics.archivedTagsTotal = repository.countArchivedTags()
    statistics.archivedRolesTotal = repository.countArchivedRoles()
    statistics.archivedMembersTotal = repository.countArchivedMembers()
    statistics
  }

  def collectConfigurationUsageStatistics: ConfigurationUsageStatistics = {
    val statistics = new ConfigurationUsageStatistics
    statistics.globalConfigurationsTotal = repository.countGlobalConfigurations()
    statistics.folderConfigurationsTotal = repository.countFolderConfigurations()
    statistics
  }

  def collectApplicationUsageStatistics: ApplicationUsageStatistics = {
    val statistics = new ApplicationUsageStatistics
    statistics.applicationsTotal = repository.countApplications()
    statistics
  }

  def collectEnvironmentsUsageStatistics: EnvironmentUsageStatistics = {
    val statistics = new EnvironmentUsageStatistics
    statistics.environmentsTotal = repository.countEnvironments()
    statistics
  }

  def collectCalendarUsageStatistics: CalendarUsageStatistics = {
    val statistics = new CalendarUsageStatistics
    statistics.calendarEntriesTotal = repository.countCalendarEntries()
    statistics
  }

  def collectReportUsageStatistics: ReportUsageStatistics = {
    val statistics = new ReportUsageStatistics
    statistics.activityLogsTotal = repository.countActivityLogs()
    statistics.attributesTotal = repository.countAttributes()
    statistics.deploymentsTotal = repository.countDeployments()
    statistics.deploymentHistoriesTotal = repository.countDeploymentHistories()
    statistics.deploymentRecordsTotal = repository.countDeploymentRecords()
    statistics.itsmRecordsTotal = repository.countItsmRecords()
    statistics.complianceRecordsTotal = repository.countComplianceRecords()
    statistics.buildRecordsTotal = repository.countBuildRecords()
    statistics.planRecordsTotal = repository.countPlanRecords()
    statistics.archivedDeploymentRecordsTotal = repository.countArchivedDeploymentRecords()
    statistics.archivedItsmRecordsTotal = repository.countArchivedItsmRecords()
    statistics.archivedComplianceRecordsTotal = repository.countArchivedComplianceRecords()
    statistics.archivedBuildRecordsTotal = repository.countArchivedBuildRecords()
    statistics.archivedPlanRecordsTotal = repository.countArchivedPlanRecords()
    statistics.permissionSnapshotsTotal = repository.countPermissionSnapshots()
    statistics.riskAssessmentsTotal = repository.countRiskAssessments()
    statistics
  }

  private def calculateAutomationPercentage(automatedTasks: Int, totalTasks: Int): Float = {
    if (totalTasks > 0) (BigDecimal(automatedTasks) / BigDecimal(totalTasks)).setScale(2, RoundingMode.HALF_UP).floatValue else 0F
  }

}

class ReleaseUsageStatistics {
  @BeanProperty
  var plannedReleasesTotal: Int = 0

  @BeanProperty
  var inProgressReleasesTotal: Int = 0

  @BeanProperty
  var pausedReleasesTotal: Int = 0

  @BeanProperty
  var failingReleasesTotal: Int = 0

  @BeanProperty
  var failedReleasesTotal: Int = 0

  @BeanProperty
  var completedReleasesTotal: Int = 0

  @BeanProperty
  var abortedReleasesTotal: Int = 0

  @BeanProperty
  var releases24HoursTotal: Int = 0

  @BeanProperty
  var releases30DaysTotal: Int = 0

  @BeanProperty
  var completionRateLastMonth: Float = 0

  @BeanProperty
  var averageReleasesPerDayLastYear: Float = 0

  @BeanProperty
  var averageReleasesPerMonthLastYear: Float = 0

  @BeanProperty
  var releasesWithDependencies: Int = 0

  @BeanProperty
  var dependenciesTotal: Int = 0

  @BeanProperty
  var averageDependenciesPerReleases: Float = 0

  @BeanProperty
  var maxDependenciesInRelease: Int = 0

  @BeanProperty
  var releaseTagsTotal: Int = 0

}

class TaskUsageStatistics {
  @BeanProperty
  var plannedTasksTotal: Int = 0

  @BeanProperty
  var pendingTasksTotal: Int = 0

  @BeanProperty
  var inProgressTasksTotal: Int = 0

  @BeanProperty
  var completedTasksTotal: Int = 0

  @BeanProperty
  var skippedTasksTotal: Int = 0

  @BeanProperty
  var failedTasksTotal: Int = 0

  @BeanProperty
  var abortedTasksTotal: Int = 0

  @BeanProperty
  var templateTasksTotal: Int = 0

  @BeanProperty
  var averageTasksPerRelease: Float = 0

  @BeanProperty
  var maxTasksInRelease: Int = 0

  @BeanProperty
  var taskTypeOccurrences: util.List[String] = new util.ArrayList[String]()

  @BeanProperty
  var automationPercentage: Float = 0

  @BeanProperty
  var taskTagsTotal: Int = 0

  @BeanProperty
  var taskBackupsTotal: Int = 0

  @BeanProperty
  var commentsTotal: Int = 0

  @BeanProperty
  var attachmentsTotal: Int = 0
}

class UserUsageStatistics {
  @BeanProperty
  var usersTotal: Int = 0

  @BeanProperty
  var externalUsersTotal: Int = 0

  @BeanProperty
  var internalUsersTotal: Int = 0

  @BeanProperty
  var enabledUsersTotal: Int = 0

  @BeanProperty
  var activeUsersLastDay: Int = 0

  @BeanProperty
  var activeUsersLast30Days: Int = 0

  @BeanProperty
  var activeUsersLast90Days: Int = 0

  @BeanProperty
  var globalRolesTotal: Int = 0

  @BeanProperty
  var teamsTotal: Int = 0

  @BeanProperty
  var roleRolesTotal: Int = 0

  @BeanProperty
  var rolePrincipalsTotal: Int = 0

  @BeanProperty
  var rolePermissionsTotal: Int = 0
}

class TemplateUsageStatistics {
  @BeanProperty
  var templatesTotal: Int = 0

  @BeanProperty
  var templateRevisionsTotal: Int = 0
}

class TriggerUsageStatistics {
  @BeanProperty
  var triggersTotal: Int = 0

  @BeanProperty
  var activeTriggersTotal: Int = 0
}

class VariableUsageStatistics {
  @BeanProperty
  var globalVariablesTotal: Int = 0

  @BeanProperty
  var folderVariablesTotal: Int = 0

  @BeanProperty
  var averageVariablesPerFolder: Float = 0
}

class DashboardUsageStatistics {
  @BeanProperty
  var globalDashboardsTotal: Int = 0

  @BeanProperty
  var folderDashboardsTotal: Int = 0
}

class FolderUsageStatistics {
  @BeanProperty
  var foldersTotal: Int = 0

  @BeanProperty
  var maxDepthOfFolder: Int = 0
}

class ArchiveUsageStatistics {
  @BeanProperty
  var archivingEnabled: Boolean = false

  @BeanProperty
  var archivingAge: Int = 0

  @BeanProperty
  var archivedReleasesTotal: Int = 0

  @BeanProperty
  var prearchivedReleasesTotal: Int = 0

  @BeanProperty
  var oldestArchivedRelease: String = _

  @BeanProperty
  var archivedCompletedReleasesTotal: Int = 0

  @BeanProperty
  var archivedAbortedReleasesTotal: Int = 0

  @BeanProperty
  var archivedPhasesTotal: Int = 0

  @BeanProperty
  var archivedTasksTotal: Int = 0

  @BeanProperty
  var archivedAttachmentsTotal: Int = 0

  @BeanProperty
  var archivedTagsTotal: Int = 0

  @BeanProperty
  var archivedRolesTotal: Int = 0

  @BeanProperty
  var archivedMembersTotal: Int = 0
}

class DeliveryUsageStatistics {
  @BeanProperty
  var patternsTotal: Int = 0

  @BeanProperty
  var deliveriesTotal: Int = 0
}

class ReleaseGroupUsageStatistics {
  @BeanProperty
  var groupsTotal: Int = 0
}

class ReportUsageStatistics {
  @BeanProperty
  var activityLogsTotal: Int = 0

  @BeanProperty
  var attributesTotal: Int = 0

  @BeanProperty
  var deploymentsTotal: Int = 0

  @BeanProperty
  var deploymentHistoriesTotal: Int = 0

  @BeanProperty
  var deploymentRecordsTotal: Int = 0

  @BeanProperty
  var itsmRecordsTotal: Int = 0

  @BeanProperty
  var complianceRecordsTotal: Int = 0

  @BeanProperty
  var buildRecordsTotal: Int = 0

  @BeanProperty
  var planRecordsTotal: Int = 0

  @BeanProperty
  var archivedDeploymentRecordsTotal: Int = 0

  @BeanProperty
  var archivedItsmRecordsTotal: Int = 0

  @BeanProperty
  var archivedComplianceRecordsTotal: Int = 0

  @BeanProperty
  var archivedBuildRecordsTotal: Int = 0

  @BeanProperty
  var archivedPlanRecordsTotal: Int = 0

  @BeanProperty
  var permissionSnapshotsTotal: Int = 0

  @BeanProperty
  var riskAssessmentsTotal: Int = 0
}

class ConfigurationUsageStatistics {
  @BeanProperty
  var globalConfigurationsTotal: Int = 0

  @BeanProperty
  var folderConfigurationsTotal: Int = 0
}

class ApplicationUsageStatistics {
  @BeanProperty
  var applicationsTotal: Int = 0
}

class EnvironmentUsageStatistics {
  @BeanProperty
  var environmentsTotal: Int = 0
}

class CalendarUsageStatistics {
  @BeanProperty
  var calendarEntriesTotal: Int = 0
}
