package com.xebialabs.xlrelease.support

import com.xebialabs.deployit.checks.Checks.checkTrue
import com.xebialabs.deployit.repository.{WorkDirContext, WorkDirFactory}
import com.xebialabs.deployit.security.permission.PlatformPermissions.ADMIN
import com.xebialabs.xlplatform.support.api.SupportService
import com.xebialabs.xlplatform.support.rest.{SupportAcceleratorConfiguration, SupportResource}
import com.xebialabs.xlrelease.api.utils.ResponseHelper
import com.xebialabs.xlrelease.domain.status.TaskStatus
import com.xebialabs.xlrelease.repository.Ids
import com.xebialabs.xlrelease.repository.Ids.{isReleaseId, isTaskId, releaseIdFrom}
import com.xebialabs.xlrelease.support.report.{SystemInformationService, TaskManagerInformationService}
import com.xebialabs.xlrelease.support.tools.SupportUtilities
import grizzled.slf4j.Logging
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Controller

import java.io.OutputStream
import javax.ws.rs._
import javax.ws.rs.core._
import scala.util.Try

@Controller
@Path("/support")
class XlrSupportResource @Autowired()(supportService: SupportService,
                                      supportUtilities: SupportUtilities,
                                      systemInformationService: SystemInformationService,
                                      supportAcceleratorConfiguration: SupportAcceleratorConfiguration,
                                      taskManagerInformationService: TaskManagerInformationService)
  extends SupportResource(supportService, supportAcceleratorConfiguration) with Logging {

  override def generateSupportZip(): Response = {
    checkPermission(ADMIN)

    Try(systemInformationService.writeSystemInformationReport())
      .recover { case exception => logger.warn("Unable to generate system information report due to error", exception) }
    Try(systemInformationService.writeThreadDump())
      .recover { case exception => logger.warn("Unable to generate thread dump due to error", exception) }
    Try(taskManagerInformationService.writeTaskManagerReport())
      .recover { case exception => logger.warn("Unable to generate task manager information report due to error", exception) }

    super.generateSupportZip()
  }

  @GET
  @Path("/system-information")
  @Produces(Array(MediaType.APPLICATION_JSON, MediaType.TEXT_HTML))
  def getSystemInformation(@QueryParam("reports") reports: String, @Context request: Request): Response = {
    checkPermission(ADMIN)

    val requestedReports = if (reports != null) reports.toLowerCase.split(",").toSeq else Seq.empty
    if (acceptsJson(request)) {
      Response.ok(systemInformationService.generateJsonReport(requestedReports), MediaType.APPLICATION_JSON).build
    } else {
      Response.ok(systemInformationService.generateHtmlReport(requestedReports), MediaType.TEXT_HTML).build
    }
  }

  def acceptsJson(request: Request): Boolean = {
    val variants = Variant.mediaTypes(MediaType.APPLICATION_JSON_TYPE, MediaType.TEXT_HTML_TYPE).build()
    val variant = request.selectVariant(variants)
    variant != null && variant.getMediaType.isCompatible(MediaType.APPLICATION_JSON_TYPE)
  }

  @GET
  @Path("/releases/{releaseId:.*Release[^/]*}")
  @Produces(Array(MediaType.APPLICATION_OCTET_STREAM))
  def downloadRelease(@PathParam("releaseId") releaseId: String): Response = {
    checkPermission(ADMIN)

    val content = supportUtilities.getReleaseContent(releaseId)
    val output: StreamingOutput = (outputStream: OutputStream) => {
      WorkDirContext.initWorkdir(WorkDirFactory.DOWNLOAD_WORKDIR_PREFIX)
      try {
        outputStream.write(content.getBytes("UTF-8"))
      } finally {
        WorkDirContext.get().delete()
        WorkDirContext.clear()
      }
    }

    val fileName = s"${Ids.getName(releaseId)}.json"
    ResponseHelper.streamFile(fileName, output, "application/json")
  }

  @DELETE
  @Path("/releases/{releaseId:.*Release[^/]*}")
  def deleteRelease(@PathParam("releaseId") releaseId: String): Unit = {
    checkPermission(ADMIN)
    checkTrue(isReleaseId(releaseId), s"ID $releaseId is not a valid release ID")

    supportUtilities.deleteRelease(releaseId)
  }

  @POST
  @Path("/update-analytics-id")
  def updateAnalyticsId(): Unit = {
    supportUtilities.updateAnalyticsId()
  }

  @GET
  @Path("/current-analytics-id")
  def getCurrentAnalyticsId: String = {
    supportUtilities.getCurrentAnalyticsId
  }

  @POST
  @Path("/tasks/{taskId:.*Task[^/]*}")
  def updateTaskStatus(@PathParam("taskId") taskId: String,
                       @QueryParam("newStatus") newStatus: TaskStatus): Unit = {
    checkPermission(ADMIN)
    checkTrue(isTaskId(taskId), s"ID $taskId is not a valid task ID")
    checkTrue(releaseIdFrom(taskId) != null, s"ID $taskId does not contain the release ID")

    supportUtilities.updateTaskStatus(taskId, newStatus)
  }
}
