package com.xebialabs.xlrelease.script

import com.xebialabs.xlrelease.domain.status.TaskStatus
import com.xebialabs.xlrelease.script.DefaultScriptService.{CustomScriptTaskResults, ScriptTaskResults}
import com.xebialabs.xlrelease.service.RestartPhasesException
import grizzled.slf4j.Logging
import org.springframework.security.core.Authentication

import java.util
import java.util.Date

sealed trait TaskResult

sealed trait ScriptResult extends Logging {
  def executionLog: String

  def attachmentId: Option[String]

  def authentication: Authentication

  def taskId: String

  def executionId: String
}

trait ScriptTaskResult extends ScriptResult

trait PreconditionResult extends ScriptResult

case class ValidPreconditionResult(taskId: String,
                                   executionId: String,
                                   executionLog: String,
                                   attachmentId: Option[String],
                                   authentication: Authentication) extends PreconditionResult

case class ExceptionPreconditionResult(taskId: String,
                                       executionId: String,
                                       executionLog: String,
                                       attachmentId: Option[String],
                                       authentication: Authentication) extends PreconditionResult

case class InvalidPreconditionResult(taskId: String,
                                     executionId: String,
                                     executionLog: String,
                                     attachmentId: Option[String],
                                     authentication: Authentication) extends PreconditionResult


case class SuccessScriptTaskResult(taskId: String,
                                   executionId: String,
                                   executionLog: String,
                                   attachmentId: Option[String],
                                   scriptTaskResults: ScriptTaskResults,
                                   authentication: Authentication) extends ScriptTaskResult

case class RestartScriptTaskResult(taskId: String,
                                   executionId: String,
                                   executionLog: String,
                                   attachmentId: Option[String],
                                   scriptTaskResults: ScriptTaskResults,
                                   exception: RestartPhasesException,
                                   authentication: Authentication) extends ScriptTaskResult

case class FailureScriptTaskResult(taskId: String,
                                   executionId: String,
                                   executionLog: String,
                                   attachmentId: Option[String],
                                   scriptTaskResults: ScriptTaskResults,
                                   authentication: Authentication) extends ScriptTaskResult

sealed trait CustomScriptTaskResult extends ScriptResult

case class SuccessCustomScriptTaskResult(taskId: String,
                                         executionId: String,
                                         executionLog: String,
                                         attachmentId: Option[String],
                                         results: CustomScriptTaskResults,
                                         authentication: Authentication) extends CustomScriptTaskResult

case class FailureCustomScriptTaskResult(taskId: String,
                                         executionId: String,
                                         executionLog: String,
                                         attachmentId: Option[String],
                                         authentication: Authentication) extends CustomScriptTaskResult

case class RestartCustomScriptTaskResult(taskId: String,
                                         executionId: String,
                                         executionLog: String,
                                         attachmentId: Option[String],
                                         exception: RestartPhasesException,
                                         authentication: Authentication) extends CustomScriptTaskResult

trait FacetCheckResult extends ScriptResult

case class SuccessFacetCheckResult(taskId: String,
                                   executionId: String,
                                   executionLog: String,
                                   attachmentId: Option[String],
                                   authentication: Authentication) extends FacetCheckResult

case class FailureFacetCheckResult(taskId: String,
                                   executionId: String,
                                   executionLog: String,
                                   attachmentId: Option[String],
                                   authentication: Authentication) extends FacetCheckResult


case class AwaitFacetCheckResult(taskId: String,
                                 executionId: String,
                                 executionLog: String,
                                 attachmentId: Option[String],
                                 delayUntil: Date,
                                 authentication: Authentication) extends FacetCheckResult

sealed trait FailureHandlerResult

case class SuccessFailureHandlerResult(taskId: String, scriptTaskResults: ScriptTaskResults) extends FailureHandlerResult

case class FailureFailureHandlerResult(taskId: String, scriptTaskResults: ScriptTaskResults, exception: Exception) extends FailureHandlerResult


trait ReactiveTaskResult extends ScriptResult {
  def toCustomScriptTaskResult: Option[CustomScriptTaskResult]
}

case class MatchedEventResult(taskId: String,
                              executionId: String,
                              executionLog: String,
                              attachmentId: Option[String],
                              scriptTaskResults: CustomScriptTaskResults,
                              authentication: Authentication) extends ReactiveTaskResult {


  override def toCustomScriptTaskResult: Option[CustomScriptTaskResult] =
    Some(SuccessCustomScriptTaskResult(taskId, executionId, executionLog, attachmentId, scriptTaskResults, authentication))
}

case class UnmatchedEventResult(taskId: String,
                                executionId: String,
                                executionLog: String,
                                attachmentId: Option[String],
                                scriptTaskResults: CustomScriptTaskResults,
                                authentication: Authentication) extends ReactiveTaskResult {


  override def toCustomScriptTaskResult: Option[CustomScriptTaskResult] = None
}

case class FailedEventResult(taskId: String,
                             executionId: String,
                             executionLog: String,
                             attachmentId: Option[String],
                             authentication: Authentication) extends ReactiveTaskResult {

  override def toCustomScriptTaskResult: Option[CustomScriptTaskResult] = Some(FailureCustomScriptTaskResult(taskId, executionId, executionLog, attachmentId, authentication))
}

sealed trait ContainerTaskResult {
  def taskId: String

  def targetStatus: TaskStatus

  def outputProperties: util.Map[String, AnyRef]

  def comment: String

}

case class SuccessContainerTaskResult(taskId: String, outputProperties: util.Map[String, AnyRef], comment: String = "Completed") extends ContainerTaskResult {
  override def targetStatus: TaskStatus = TaskStatus.COMPLETED
}

case class FailureContainerTaskResult(taskId: String, outputProperties: util.Map[String, AnyRef], reason: String) extends ContainerTaskResult {
  override def targetStatus: TaskStatus = TaskStatus.FAILED

  def comment: String = {
    s"Execution failed.${System.lineSeparator()}Cause: $reason"
  }
}
// END
