package com.xebialabs.xlrelease.service

import com.codahale.metrics.annotation.Timed
import com.xebialabs.platform.script.jython.ScriptSource
import com.xebialabs.platform.script.jython.Syntactic.wrapperCodeWithLib
import com.xebialabs.xlrelease.domain.{Configuration, ExternalVariableServer, ScriptHelper}
import com.xebialabs.xlrelease.script.EncryptionHelper.decrypt
import com.xebialabs.xlrelease.views.SharedConfigurationStatusResponse
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.{Component, Service}

import java.io.{FileNotFoundException, StringWriter}
import java.util.{List => JList}
import scala.collection.immutable.HashMap
import scala.jdk.OptionConverters._
import scala.util.Try

@Service
class SharedConfigurationStatusService @Autowired()(configurationStatusServices: JList[ConfigurationStatusService],
                                                    configurationVariableService: ConfigurationVariableService) {

  @Timed
  def hasScript(configuration: Configuration): Boolean = {
    getApplicableService(configuration).isDefined
  }

  @Timed
  def checkStatus(configuration: Configuration): SharedConfigurationStatusResponse = {
    getApplicableService(configuration) match {
      case Some(service) =>
        if (!configuration.isInstanceOf[ExternalVariableServer]) {
          configurationVariableService.resolve(configuration)
        }
        decrypt(configuration)
        service.checkStatus(configuration)
      case None =>
        val msg = s"Configuration '${configuration.getTitle}' of type '${configuration.getType}' does not have status check logic implemented."
        new SharedConfigurationStatusResponse(false, false, msg)
    }
  }

  def getApplicableService(configuration: Configuration): Option[ConfigurationStatusService] = {
    configurationStatusServices.stream().filter(_.supports(configuration)).findFirst().asScala
  }
}

trait ConfigurationStatusService {
  def supports(configuration: Configuration): Boolean

  def checkStatus(configuration: Configuration): SharedConfigurationStatusResponse
}

@Component
class DefaultConfigurationStatusService @Autowired()()
  extends ConfigurationStatusService with BaseJythonSupport[Configuration] {

  override def supports(configuration: Configuration): Boolean = {
    Try(ScriptHelper.getScript(configuration)).isSuccess
  }

  def checkStatus(configuration: Configuration): SharedConfigurationStatusResponse = {
    val outputStringWriter = new StringWriter()
    val errorStringWriter = new StringWriter()

    val libraries = Seq(
      ScriptSource.byResource("xlrelease/OAuthSupport.py"),
      ScriptSource.byResource("xlrelease/HttpRequest.py"),
      ScriptSource.byResource("xlrelease/HttpResponse.py")
    )
    val ctxAttrs = HashMap("configuration" -> configuration)

    try {
      execute(configuration, libraries ++ wrapperCodeWithLib(Seq("configuration")), ctxAttrs, outputStringWriter, errorStringWriter)
      new SharedConfigurationStatusResponse(true, true)
    } catch {
      case e: FileNotFoundException =>
        logger.error(s"There was an error reading the script file", e)
        new SharedConfigurationStatusResponse(false, false)
      case e: Exception =>
        logger.error(s"There was an exception while executing script", e)
        new SharedConfigurationStatusResponse(false, true,
          if (e.getMessage != null) {
            e.getMessage.substring(0, e.getMessage.indexOf("in <script>")).replace("Error while executing script [jython-expression]:", "")
          } else {
            null
          })
    } finally {
      logger.debug(
        s"""Script output :
           |${outputStringWriter.toString}""".stripMargin)

      logger.debug(
        s"""Script errors :
           |${errorStringWriter.toString}""".stripMargin)
    }
  }

}
