package com.xebialabs.xlrelease.json

import com.fasterxml.jackson.databind.JsonNode
import com.fasterxml.jackson.databind.node.ObjectNode
import com.xebialabs.deployit.plugin.api.reflect.{PropertyKind, Type}
import com.xebialabs.xlrelease.domain.status.FlagStatus
import com.xebialabs.xlrelease.export.TemplateJsonHelper.{flagTask, getPhase, typeNotFound}
import grizzled.slf4j.Logging

import scala.collection.mutable.ListBuffer
import scala.jdk.CollectionConverters._

object JsonKeys {
  val variables = "variables"
  val valueProvider = "valueProvider"
  val status = "status"
  val tiles = "tiles"
  val extensions = "extensions"
  val facets = "facets"
  val pythonScript = "pythonScript"
}

object JsonReplacements extends Logging {

  val unknownTypeName = "xlrelease.UnknownType"
  val originalType = "originalType"
  val defaultProperties: Set[String] = Set("id", "type", originalType, "backup", "$createdBy", "$createdAt", "$lastModifiedBy", "$lastModifiedAt", "$token",
    "$scmTraceabilityDataId")

  def typeFound(typeAlias: String): Boolean = Type.valueOf(typeAlias).exists

  implicit class PythonScriptOps(val pythonScript: ObjectNode) extends AnyVal {

    def isUnknownTypeRegistered: Boolean = pythonScript.get("type").asText == unknownTypeName && typeFound(pythonScript.get(originalType).asText)

    def replaceNotFoundTask(task: ObjectNode, maybeRelease: Option[ObjectNode] = None): Unit = {
      val missingType = pythonScript.get("type").asText
      val backup = pythonScript.toString
      logger.debug(s"Missing type found $missingType, backing up $backup for task with id ${task.get("id").asText}")
      pythonScript.put("backup", backup)
      pythonScript.put("originalType", missingType)

      val unknownType = Type.valueOf(unknownTypeName)
      val originalType = pythonScript.get("type").asText
      pythonScript.put("type", unknownType.toString)

      val propertiesOnMissingTask = unknownType.getDescriptor.getPropertyDescriptors.asScala.map(_.getName).toSet
      val propertiesOnUnknownTask = keyExtractor(pythonScript)
      propertiesOnUnknownTask
        .filter(p => !propertiesOnMissingTask.contains(p) && !defaultProperties.contains(p))
        .foreach(pythonScript.remove)

      val maybePhase = maybeRelease.map { release: ObjectNode => getPhase(task, release) }
      val warning = s"Task '${task.get("title").asText}' in Phase '${maybePhase.fold("Not defined") { phase => phase.get("title").asText }}' has been " +
        s"replaced by an unknown task. The task of type '$originalType' could not be found because of a missing plugin."
      flagTask(task, warning)
    }

    def recoverUnknownTask(task: ObjectNode): Unit = {
      val backup: ObjectNode = JsonUtils.objectMapper.createObjectNode()
      val backupObject = JsonUtils.objectMapper.readTree(pythonScript.get("backup").asText).asInstanceOf[ObjectNode]
      backup.setAll(backupObject)
      backup.set("customScriptTask", pythonScript.get("customScriptTask"))
      backup.set("id", pythonScript.get("id"))
      task.set(JsonKeys.pythonScript, backup)
      task.put("flagStatus", FlagStatus.OK.toString)
      task.put("flagComment", "")
    }

    def ignoreTaskUnknownProperties(task: ObjectNode, maybeRelease: Option[ObjectNode] = None): Unit = {
      val registeredType = Type.valueOf(pythonScript.get("type").asText)
      val typeProps = registeredType.getDescriptor.getPropertyDescriptors.asScala.map(_.getName).toList
      val jsonProps = keyExtractor(pythonScript)
      val unknownProps = jsonProps diff (typeProps concat defaultProperties.toList)
      if (unknownProps.nonEmpty) {
        val maybePhase = maybeRelease.map { release: ObjectNode => getPhase(task, release) }
        val warning = s"Task '${task.get("title").asText}' in Phase '${maybePhase.fold("Not defined") { phase => phase.get("title").asText }}' has " +
          s"properties that do not match with the known type. Following properties do not match: [${unknownProps.mkString(", ")}]."
        flagTask(task, warning)
        unknownProps.foreach(pythonScript.remove)
      }
    }

    def ignoreUnknownEnumValues(task: ObjectNode, maybeRelease: Option[ObjectNode] = None): Unit = {
      val registeredType = Type.valueOf(pythonScript.get("type").asText)
      val enumProps = registeredType.getDescriptor.getPropertyDescriptors.asScala.filter(_.getKind == PropertyKind.ENUM).toList
      val unknownEnumValuesProps = ListBuffer.empty[String]
      enumProps.foreach { prop =>
        val enumValues = registeredType.getDescriptor.getPropertyDescriptor(prop.getName).getEnumValues
        if (pythonScript.has(prop.getName) && !enumValues.contains(pythonScript.get(prop.getName).asText)) {
          unknownEnumValuesProps += prop.getName
        }
      }
      if (unknownEnumValuesProps.nonEmpty) {
        val maybePhase = maybeRelease.map { release: ObjectNode => getPhase(task, release) }
        val warning = s"Task '${task.get("title").asText}' in Phase '${maybePhase.fold("Not defined") { phase => phase.get("title").asText }}' " +
          s"has value that do not match with possible enum values. Following properties do not match: [${unknownEnumValuesProps.mkString(", ")}]."
        flagTask(task, warning)
        unknownEnumValuesProps.foreach(pythonScript.remove)
      }
    }

  }

  val systemProps = List("id", "type")

  implicit class JSONArrayOps(val obj: ObjectNode) extends AnyVal {

    def toSeq(key: String): Seq[JsonNode] = {
      if (obj.has(key)) {
        obj.get(key).elements().asScala.toList
      } else {
        Seq()
      }
    }

    def fixJsonArray(key: String): Unit = {
      val originalArray = obj.toSeq(key)
      originalArray.foreach(originalNode => originalNode.asInstanceOf[ObjectNode].fixProperties())
      val filtered = originalArray.filter { obj => typeFound(obj.get("type").asText) }
      if (filtered.length < originalArray.length) {
        (originalArray diff filtered).foreach { notFound =>
          logger.info(s"Missing type found ${notFound.get("type").asText}, ignoring type with id ${notFound.get("id").asText}")
        }
        obj.putArray(key).addAll(filtered.asJava)
      }
    }
  }

  private val standardDashboard = "xlrelease.Dashboard"

  private def assumingDashboard(extension: JsonNode) =
    extension.has(JsonKeys.tiles) && extension.has("rows") && extension.has("columns")

  def keyExtractor(jsonNode: JsonNode): List[String] = {
    jsonNode.fields.asScala.map(_.getKey).toList
  }

  implicit class JSONObjectOps(val parent: ObjectNode) extends AnyVal {
    def fixInnerType(key: String): Unit = {
      if (parent.has(key) && parent.get(key).isInstanceOf[ObjectNode]) {
        val objectNode = parent.get(key).asInstanceOf[ObjectNode]
        val typeName = objectNode.get("type").asText
        if (!typeFound(typeName)) {
          parent.remove(key)
        } else {
          val typeProps = Type.valueOf(typeName).getDescriptor.getPropertyDescriptors.asScala.map(_.getName).toList
          val jsonProps = keyExtractor(objectNode)
          (jsonProps diff (typeProps concat systemProps)).foreach(objectNode.remove)
        }
      }
    }

    def fixProperties(): Unit = {
      val typeName = parent.get("type").asText
      if (typeFound(typeName)) {
        val typeInSystem = Type.valueOf(typeName)
        val typeProps = typeInSystem.getDescriptor.getPropertyDescriptors.asScala.map(_.getName).toList
        val jsonProps = keyExtractor(parent)
        (jsonProps diff (typeProps concat systemProps)).foreach(parent.remove)
      }
    }

    def fixDashboard(): Unit = {
      if (assumingDashboard(parent)) {
        val dashboardType = parent.get("type").asText
        if (!typeFound(dashboardType)) {
          logger.debug(s"Missing dashboard type found $dashboardType, replacing with $standardDashboard type. Dashboard id ${parent.get("id")}")
          parent.put("type", standardDashboard)
        }
        parent.fixJsonArray(JsonKeys.tiles)
      }
    }

    def fixTask(maybeRelease: Option[ObjectNode] = None): Unit = {
      if (parent.has(JsonKeys.pythonScript)) {
        parent.get(JsonKeys.pythonScript).asInstanceOf[ObjectNode] match {
          case pythonScript if typeNotFound(pythonScript) => pythonScript.replaceNotFoundTask(parent, maybeRelease)
          case pythonScript if pythonScript.isUnknownTypeRegistered => pythonScript.recoverUnknownTask(parent)
          case pythonScript =>
            pythonScript.ignoreTaskUnknownProperties(parent, maybeRelease)
            pythonScript.ignoreUnknownEnumValues(parent, maybeRelease)
        }
      }
    }
  }

}
