package com.xebialabs.xlrelease.scheduler

import com.xebialabs.xlrelease.scheduler.domain.LocalJobRunner
import com.xebialabs.xlrelease.scheduler.workers.CompositeWorker
import grizzled.slf4j.Logging

import java.util.concurrent.atomic.AtomicBoolean
import scala.util.Try

class WorkerThread(jobQueue: JobQueue, compositeWorker: CompositeWorker) extends Runnable with Logging {
  private val isRunning = new AtomicBoolean(true)

  def stop(): Unit = {
    isRunning.set(false)
    jobQueue.submit(StopWorkerThread())
  }

  // scalastyle:off cyclomatic.complexity
  override def run(): Unit = {
    while (isRunning.get()) {
      try {
        val possibleJob = jobQueue.get(LocalJobRunner.getId())
        possibleJob.foreach {
          case StopWorkerThread() =>
            logger.debug("Thread is stopped because stop job was received")
            isRunning.set(false)
            ()
          case job =>
            val isConfirmed = jobQueue.confirm(runnerId = LocalJobRunner.getId(), job.getId())
            val jobInfo: String = logFriendlyJobInfo(job)

            if (isConfirmed) {
              val result = Try(compositeWorker.execute(job))
              result.recover { case e =>
                logger.error(s"Unhandled exception while executing $jobInfo. Please report a defect.", e)
              }

              Try(jobQueue.finish(job.id)).recover { case e =>
                logger.error(s"Unable to finish $jobInfo", e)
              }

              Try(result.foreach(compositeWorker.processResult)).recover { case e =>
                logger.error(s"Unhandled exception when processing result of $jobInfo", e)
              }
            } else {
              logger.warn(s"Execution could not be confirmed for $jobInfo. Execution was skipped.")
            }
        }
      } catch {
        case _: JobQueueStopped =>
          logger.info("Thread is being stopped because job queue is stopped")
          isRunning.set(false)
        case _: InterruptedException =>
          logger.info("Thread is being terminated by a shutdown")
          isRunning.set(false)
        case e: Exception => logger.error(s"Unhandled exception", e)
      }
    }
  }

  private def logFriendlyJobInfo(job: Job): String = {
    val info = job match {
      case taskJob: TaskJob[_] => s"a job [${job.jobType}] with jobId [${job.id}] of task [${taskJob.taskId}]"
      case _ => s"a job [${job.jobType}] with jobId [${job.id}]"
    }
    info
  }
}
