package com.xebialabs.xlrelease.views;

import java.util.List;
import com.fasterxml.jackson.databind.annotation.JsonDeserialize;
import com.fasterxml.jackson.databind.annotation.JsonSerialize;

import com.xebialabs.xlrelease.api.v1.views.TeamMemberView;
import com.xebialabs.xlrelease.domain.Team;
import com.xebialabs.xlrelease.serialization.json.jackson.DomainIdDeserializer;
import com.xebialabs.xlrelease.serialization.json.jackson.DomainIdSerializer;
import com.xebialabs.xlrelease.views.converters.TeamMemberViewConverter;

import static com.google.common.collect.Lists.newArrayList;
import static com.xebialabs.xlrelease.api.v1.views.TeamMemberView.MemberType.PRINCIPAL;
import static com.xebialabs.xlrelease.api.v1.views.TeamMemberView.MemberType.ROLE;
import static com.xebialabs.xlrelease.builder.TeamBuilder.newTeam;
import static java.util.stream.Collectors.toList;

public class TeamView {

    @JsonSerialize(using = DomainIdSerializer.class)
    @JsonDeserialize(using = DomainIdDeserializer.class)
    private String id;
    private String teamName;
    private List<TeamMemberView> members = newArrayList();
    private List<String> permissions = newArrayList();
    private boolean systemTeam;

    public TeamView() {
    }

    public TeamView(Team team, TeamMemberViewConverter teamMemberViewConverter) {
        this.id = team.getId();
        this.teamName = team.getTeamName();
        this.members.addAll(team.getMembers().stream().map(t -> teamMemberViewConverter.toView(t, PRINCIPAL)).collect(toList()));
        this.members.addAll(team.getRoles().stream().map(role -> teamMemberViewConverter.toView(role, ROLE)).collect(toList()));
        this.permissions = team.getPermissions();
        this.systemTeam = team.isSystemTeam();
    }

    public String getId() {
        return id;
    }

    public String getTeamName() {
        return teamName;
    }

    public List<TeamMemberView> getMembers() {
        return members;
    }

    public List<String> getPermissions() {
        return permissions;
    }

    public boolean isSystemTeam() {
        return systemTeam;
    }

    public Team toTeam() {
        // NB: systemTeam is ignored since it's a computed property
        List<String> principals = members.stream().filter(member -> member.getType() == PRINCIPAL).map(TeamMemberView::getName).collect(toList());
        List<String> roles = members.stream().filter(member -> member.getType() == ROLE).map(TeamMemberView::getName).collect(toList());
        return newTeam().withId(id).withTeamName(teamName).withMembers(principals).withRoles(roles).withPermissions(permissions).build();
    }
}
