package com.xebialabs.xlrelease.scheduler

import com.xebialabs.deployit.exception.NotFoundException
import com.xebialabs.xlrelease.actors.ReleaseActorService
import com.xebialabs.xlrelease.domain.runner.JobRunner
import com.xebialabs.xlrelease.repository.JobRunnerRepository
import com.xebialabs.xlrelease.runner.domain.{JobId, RunnerId}
import com.xebialabs.xlrelease.scheduler.domain.LocalJobRunner
import grizzled.slf4j.Logging
import org.springframework.context.annotation.Primary
import org.springframework.dao.OptimisticLockingFailureException
import org.springframework.stereotype.Component

import java.util.concurrent.ConcurrentHashMap
import java.util.function.Predicate
import scala.annotation.tailrec
import scala.jdk.CollectionConverters._
import scala.util.{Failure, Success, Try}


trait RunnerRegistry {
  def registerJobRunner(jobRunner: JobRunner): Unit

  def unregisterJobRunner(jobRunner: JobRunner): Unit
}

trait JobProvider {
  def get(jobRunnerId: RunnerId): Option[Job]
}

@Primary
@Component
class CapabilityAwareJobQueue(jobQueue: DefaultJobQueue,
                              val jobRunnerRepository: JobRunnerRepository,
                              releaseActorService: ReleaseActorService,
                              backpressure: ScriptBackpressure)
  extends JobQueue with RunnerRegistry with JobProvider with Logging {

  private val runnerQueues: ConcurrentHashMap[RunnerId, JobRunnerQueue] = new ConcurrentHashMap[RunnerId, JobRunnerQueue]()

  createRunnerQueue(LocalJobRunner) // TODO remember to check if we have to create this one inside start and remove it inside stop ...

  override def registerJobRunner(jobRunner: JobRunner): Unit = {
    logger.debug(s"Going to register runner[${jobRunner.getId}]")
    createRunnerQueue(jobRunner)
  }

  override def unregisterJobRunner(jobRunner: JobRunner): Unit = {
    logger.debug(s"Going to un-register runner[${jobRunner.getId}]")
    removeRunnerQueue(jobRunner)
  }

  @tailrec
  override final def get(runnerId: RunnerId): Option[Job] = {
    val s = Option(runnerQueues.get(runnerId))
    s match {
      case Some(runnerQueue) => Option(runnerQueue.poll()) match {
        case msg@Some(StopWorkerThread()) => msg
        case Some(possibleJob) =>
          backpressure.backpressureFn(possibleJob)
          reserveJob(possibleJob, runnerId) match {
            case None => get(runnerId)
            case job@Some(_) => job
          }
        case None => None
      }
      case None =>
        val msg = s"No runner queue found for $runnerId"
        logger.warn(msg)
        throw new NotFoundException(msg)
    }
  }

  private def reserveJob(possibleJob: Job, runnerId: RunnerId): Option[Job] = {
    Try(reserve(possibleJob, runnerId)) match {
      case Success(reservedJob) =>
        logger.debug(s"giving $reservedJob")
        updateTaskStatusLine(reservedJob)
        Some(reservedJob)
      case Failure(exception) =>
        exception match {
          case _: OptimisticLockingFailureException =>
            logger.debug(s"Unable to reserve job $possibleJob")
            None
          case t: Throwable =>
            logger.error("Unexpected error", t)
            // there is no need to send this to the runner - we want to fail this job
            val failJob = FailJob(possibleJob, Some(t.getMessage))
            // it should eventually (in the future) execute fail using local runner so it's not stored in the database
            runnerQueues.get(LocalJobRunner.getId()).offer(failJob)
            None
        }
    }
  }

  private def reserve(possibleJob: Job, runnerId: RunnerId): Job = {
    val job = jobQueue.reserve(possibleJob, runnerId)
    removeFromJobRunnerQueues(_.id == possibleJob.id)
    job
  }

  override def confirm(runnerId: RunnerId, jobId: JobId): Boolean = {
    val (result, taskJob) = jobQueue.confirm(runnerId, jobId)
    if (result && taskJob.isDefined) {
      updateTaskStatusLine(taskJob.get)
    }
    result
  }

  private def offerToJobRunnerQueues(job: Job): Unit = {
    for (subscriber <- runnerQueues.values().iterator().asScala) {
      subscriber.offer(job)
    }
  }

  private def removeFromJobRunnerQueues(predicate: Predicate[Job]): Unit = {
    for (subscriber <- runnerQueues.values().iterator().asScala) {
      subscriber.cancelIf(predicate)
    }
  }

  override def submit(job: Job): Unit = {
    jobQueue.submit(job)
    updateTaskStatusLine(job)
    offerToJobRunnerQueues(job)
  }

  override def submitExisting(job: Job): Unit = {
    jobQueue.submitExisting(job)
    offerToJobRunnerQueues(job)
  }

  override def submitBroadcasted(job: Job): Unit = {
    jobQueue.submitBroadcasted(job)
    offerToJobRunnerQueues(job)
  }

  override def replace(job: Job): Unit = {
    jobQueue.replace(job)
    offerToJobRunnerQueues(job)
  }

  override def finish(jobId: JobId): Unit = {
    jobQueue.finish(jobId)
  }

  override def cancelIf(predicate: Predicate[Job]): Unit = {
    jobQueue.cancelIf(predicate)
    removeFromJobRunnerQueues(predicate)
  }

  override def start(): Unit = {
    registerJobRunners()
    jobQueue.start()
  }

  override def stop(): Unit = {
    jobQueue.stop()
  }

  override def isRunning(): Boolean = {
    jobQueue.isRunning()
  }

  override def size(): Int = {
    jobQueue.size()
  }

  private def registerJobRunners(): Unit = {
    val jobRunners = jobRunnerRepository.findAll()
    jobRunners.forEach(createRunnerQueue)
  }

  private def createRunnerQueue(runner: JobRunner): Unit = {
    runnerQueues.remove(runner.getId)
    val runnerQueue = JobRunnerQueue(runner.getCapabilities.asScala.toSet)
    runnerQueues.put(runner.getId, runnerQueue)
    // offerToJobRunnerQueue and do not care about order
    for (job <- jobQueue.localJobs.iterator().asScala) {
      runnerQueue.offer(job)
    }
  }

  private def removeRunnerQueue(runner: JobRunner): Unit = {
    runnerQueues.remove(runner.getId)
  }

  private def updateTaskStatusLine(job: Job): Unit = {
    job match {
      case taskJob: TaskJob[_] =>
        taskJob.jobStatusLine().foreach { statusLine =>
          releaseActorService.updateTaskStatusLine(taskJob.taskId, statusLine)
        }
      case _ => ()
    }
  }

}

