package com.xebialabs.xlrelease.domain.tasks.container;

import java.util.*;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import com.xebialabs.deployit.plugin.api.reflect.PropertyDescriptor;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.xlrelease.domain.ContainerTask;
import com.xebialabs.xlrelease.domain.ContainerTaskDefinition;
import com.xebialabs.xlrelease.repository.ConfigurationRepository;
import com.xebialabs.xlrelease.variable.VariableViewHelper;
import com.xebialabs.xlrelease.views.TaskFullView;
import com.xebialabs.xlrelease.views.converters.BaseTaskViewConverter;

import scala.Tuple2;

import static com.xebialabs.deployit.booter.local.utils.Strings.isNotEmpty;
import static com.xebialabs.xlrelease.builder.TaskBuilder.newContainerTask;

@Component
public class ContainerTaskViewConverter extends BaseTaskViewConverter<ContainerTask> {

    private final ConfigurationRepository configurationRepository;

    @Autowired
    public ContainerTaskViewConverter(final ConfigurationRepository configurationRepository) {
        this.configurationRepository = configurationRepository;
    }

    @Override
    protected Class<ContainerTask> getTaskClass() {
        return ContainerTask.class;
    }

    @Override
    protected ContainerTask fromView(final TaskFullView view) {
        ContainerTask task = newContainerTask(view.getType())
                .withId(view.getId())
                .withStatusLine(view.getStatusLine())
                .withKeepPreviousOutputPropertiesOnRetry(view.isKeepPreviousOutputPropertiesOnRetry())
                .build();

        // common fields will be filled by BaseTaskViewConverter
        // there can be input / output properties...
        // potentially there can be variable mapping
        ContainerTaskDefinition containerTaskDefinition = new ContainerTaskDefinition(task.getType());
        fillProperties(view.getInputProperties(), containerTaskDefinition.getInputProperties(), task);
        fillProperties(view.getOutputProperties(), containerTaskDefinition.getOutputProperties(), task);
        task.setConfigurationUri(view.getConfigurationUri());
        return task;
    }

    @Override
    public TaskFullView toFullView(final ContainerTask task, final List<Type> allowedTaskTypesForAuthenticatedUser) {
        TaskFullView taskFullView = super.toFullView(task, allowedTaskTypesForAuthenticatedUser);
        Map<String, String> variableMapping = task.getVariableMapping();
        task.getTaskType().getDescriptor().getPropertyDescriptors().stream().filter(pd -> "input".equals(pd.getCategory())).forEach(pd ->
                taskFullView.getInputProperties().put(pd.getName(), viewValue(pd, task, variableMapping, false))
        );
        task.getTaskType().getDescriptor().getPropertyDescriptors().stream().filter(pd -> "output".equals(pd.getCategory())).forEach(pd ->
                taskFullView.getOutputProperties().put(pd.getName(), viewValue(pd, task, variableMapping, true))
        );
        taskFullView.setOutputVarHelpText(task.getOutputVarHelpText());
        taskFullView.setColor(task.getTaskColor());
        taskFullView.setCustomIconClass(task.getIconClass());
        taskFullView.setCustomIconLocation(task.getIconLocation());
        taskFullView.setScriptDefinitionType(task.getType().toString());
        taskFullView.setConfigurationUri(task.getConfigurationUri());
        taskFullView.setStatusLine(task.getStatusLine());
        taskFullView.setKeepPreviousOutputPropertiesOnRetry(task.isKeepPreviousOutputPropertiesOnRetry());
        taskFullView.setContainerTask(true);
        return taskFullView;
    }

    private Object viewValue(final PropertyDescriptor pd, final ContainerTask task, final Map<String, String> variableMapping, final boolean useViewForStrings) {
        final String propertyName = pd.getName();
        return VariableViewHelper.toView(pd, variableMapping.get(propertyName), task.getProperty(propertyName), useViewForStrings);
    }

    private void fillProperties(final Map<String, Object> viewProperties, final Collection<PropertyDescriptor> propertyDescriptors, final ContainerTask containerTask) {
        propertyDescriptors.forEach( pd ->
                {
                    final String propertyName = pd.getName();
                    final Object viewValue = viewProperties.get(propertyName);
                    if (viewValue == null) {
                        containerTask.setProperty(propertyName, null);
                    } else {
                        Tuple2<String, Object> variableAndValue = VariableViewHelper.fromView(pd, viewValue, configurationRepository);
                        String variable = variableAndValue._1();
                        Object value = variableAndValue._2();
                        if (value != null) {
                            switch (pd.getKind()) {
                                case SET_OF_STRING:
                                case SET_OF_CI:
                                    value = new HashSet<>((Collection<?>) value);
                                    break;
                                case LIST_OF_CI:
                                case LIST_OF_STRING:
                                    value = new ArrayList<>((Collection<?>) value);
                                    break;
                                default:
                                    // nothing
                            }
                        }
                        containerTask.setProperty(propertyName, value);
                        updateVariableMappingOfProperty(containerTask, propertyName, variable);
                    }
                }

        );
    }

    @Override
    protected void fillVariableMappings(final TaskFullView view, final ContainerTask task) {
        // variable mappings updated by the fillProperties method
    }

    private void updateVariableMappingOfProperty(final ContainerTask containerTask, final String propertyName, final String variableName) {
        final Map<String, String> variableMapping = containerTask.getVariableMapping();
        if (isNotEmpty(variableName)) {
            variableMapping.put(propertyName, variableName);
        } else {
            variableMapping.remove(propertyName);
        }
    }
}
