package com.xebialabs.xlrelease.actors

import akka.actor._
import com.typesafe.akka.extension.quartz.QuartzSchedulerExtension
import com.xebialabs.xlplatform.cluster.NodeState.isActive
import com.xebialabs.xlrelease.config.XlrConfig
import com.xebialabs.xlrelease.service.ArchivePurgingScheduleService
import com.xebialabs.xlrelease.utils.QuartzUtils._

import scala.concurrent.Future
import scala.util.{Failure, Success}

object ArchivePurgingActor {
  def props(service: ArchivePurgingScheduleService, xlrConfig: XlrConfig) = Props(new ArchivePurgingActor(service, xlrConfig))

  def name = "archivePurging"

  def scheduleName = "archivePurgingSchedule"

  private[actors] case object Tick

  private case object JobDone

}

class ArchivePurgingActor(service: ArchivePurgingScheduleService, xlrConfig: XlrConfig) extends Actor with ActorLogging {

  private val scheduler = QuartzSchedulerExtension(context.system)
  private implicit val executionContext = context.system.dispatcher

  override def preStart() = {
    val cron = service.getCronSchedule.asQuartzCron
    scheduler.createSchedule(
      name = ArchivePurgingActor.scheduleName,
      cronExpression = cron
    )
    log.info("Scheduling archive purging job using cron " + cron)
    scheduler.schedule(ArchivePurgingActor.scheduleName, self, ArchivePurgingActor.Tick)
  }

  override def postStop(): Unit = {
    log.info("Stopping archive purging job")
    scheduler.cancelJob(ArchivePurgingActor.scheduleName)
    super.postStop()
  }

  override def receive: Actor.Receive = scheduleJob

  private def scheduleJob: Receive = {
    case ArchivePurgingActor.Tick =>
      if (isActive) {
        log.debug("Starting archive purging job")
        context.become(ignoreTicks)
        executeJob()
      }
  }

  private def ignoreTicks: Receive = {
    case ArchivePurgingActor.Tick =>
      log.debug("Tick is ignored. Archive Purging Job is in progress")
    case ArchivePurgingActor.JobDone =>
      context.become(scheduleJob)
      log.debug("ArchivePurgingActor ready");
  }

  private def executeJob(): Unit = {
    implicit val ec = xlrConfig.executors.auxiliaryExecutor.executionContext
    Future {
      service.purgeArchive()
    } onComplete {
      case Success(_) =>
        log.debug("Archive purging job done")
        self ! ArchivePurgingActor.JobDone
      case Failure(ex) =>
        log.debug("Error while calling archive purging job")
        log.error(ex, ex.getMessage)
        self ! ArchivePurgingActor.JobDone
    }
  }

}
