package com.xebialabs.xlrelease.cache.caffeine

import com.github.benmanes.caffeine.cache.{Cache, Caffeine}
import com.typesafe.config.{Config, ConfigFactory}
import com.xebialabs.xlrelease.config.XlrConfig
import com.xebialabs.xlrelease.config.XlrConfig.CacheSettings

import java.lang.management.ManagementFactory
import java.util.concurrent.TimeUnit
import javax.management.{ObjectInstance, ObjectName}

trait CaffeineCacheBuilder {
  def buildCache[K,V](cacheManagerName: String, name: String, defaultCacheConfigName: String, cacheSettings: CacheSettings): Cache[K, V] = {
    import XlrConfig._
    val defaultCacheConfig: Config = cacheSettings.config.getConfig(defaultCacheConfigName)
    val cacheConfig = cacheSettings.config.getOptionalConfig(name).getOrElse(ConfigFactory.empty()).withFallback(defaultCacheConfig)
    val builder = for {
      b <- Some(Caffeine.newBuilder())
      b <- Some(cacheConfig.getOptionalInt("initial-capacity").fold(b)(s => b.initialCapacity(s)))
      b <- Some(cacheConfig.getOptionalLong("max-size").fold(b)(s => b.maximumSize(s)))
      b <- Some(cacheConfig.getOptionalLong("max-weight").fold(b)(s => b.maximumWeight(s)))
      b <- Some(cacheConfig.getOptionalBoolean("weak-keys").fold(b)(s => if (s) b.weakKeys() else b))
      b <- Some(cacheConfig.getOptionalBoolean("weak-values").fold(b)(s => if (s) b.weakValues() else b))
      b <- Some(cacheConfig.getOptionalBoolean("soft-values").fold(b)(s => if (s) b.softValues() else b))
      // ttl is same as expireAfterAccess (i.e. an alias)
      b <- Some(cacheConfig.getOptionalDuration("ttl").fold(b)(d => b.expireAfterAccess(d.toSeconds, TimeUnit.SECONDS)))
      b <- Some(cacheConfig.getOptionalDuration("expire-after-access").fold(b)(d => b.expireAfterAccess(d.toSeconds, TimeUnit.SECONDS)))
      b <- Some(cacheConfig.getOptionalDuration("expire-after-write").fold(b)(d => b.expireAfterWrite(d.toSeconds, TimeUnit.SECONDS)))
      b <- Some(cacheConfig.getOptionalDuration("refresh-after-write").fold(b)(d => b.refreshAfterWrite(d.toSeconds, TimeUnit.SECONDS)))
      b <- Some(cacheConfig.getOptionalBoolean("record-stats").fold(b)(s => if (s) b.recordStats() else b))
    } yield b.build[K, V]()
    val rawCache = builder.get
    val wrapCache = cacheConfig.getOptionalBoolean("wrapped").getOrElse(false)
    val cache = if (wrapCache) {
      new CaffeineCacheDelegate[K, V](name, rawCache)
    } else {
      rawCache
    }
    if (cacheConfig.getOptionalBoolean("register-mbean").getOrElse(false)) {
      registerAsMbean(cacheManagerName, name, cache)
    }
    cache
  }


  private def registerAsMbean[K,V](cacheManagerName: String, name: String, cache: Cache[K, V]): ObjectInstance = {
    val server = ManagementFactory.getPlatformMBeanServer
    val beanType = "statistics"
    val objectname = new ObjectName(s"cache:type=$beanType,CacheManager=$cacheManagerName,Cache=$name")
    val statistics = new CaffeineStatsCounter(cache)
    server.registerMBean(statistics, objectname)
  }
}
