package com.xebialabs.xlrelease.scheduler.upgrade

import com.xebialabs.deployit.server.api.upgrade.{Upgrade, Version}
import com.xebialabs.xlrelease.domain._
import com.xebialabs.xlrelease.domain.status.TaskStatus
import com.xebialabs.xlrelease.repository.sql.persistence.Schema.TASKS
import com.xebialabs.xlrelease.repository.{Ids, ReleaseRepository, TaskRepository}
import com.xebialabs.xlrelease.scheduler._
import com.xebialabs.xlrelease.scheduler.repository.JobRepository
import com.xebialabs.xlrelease.script.TaskSoftReference
import com.xebialabs.xlrelease.serialization.json.repository.ResolveOptions
import com.xebialabs.xlrelease.upgrade.Components.XL_RELEASE_COMPONENT
import com.xebialabs.xlrelease.upgrade.UpgradeSupport.{BatchSupport, ParallelSupport, TransactionSupport}
import grizzled.slf4j.Logging
import org.springframework.beans.factory.annotation.{Autowired, Qualifier}
import org.springframework.jdbc.core.{JdbcTemplate, RowMapper}
import org.springframework.transaction.support.TransactionTemplate

import java.sql.ResultSet
import scala.collection.mutable
import scala.concurrent.duration.Duration
import scala.jdk.CollectionConverters._
import scala.util.Try

class QueuedTasksUpgrade @Autowired()(@Qualifier("xlrRepositoryJdbcTemplate") val jdbcTemplate: JdbcTemplate,
                                      @Qualifier("xlrRepositoryTransactionTemplate") val transactionTemplate: TransactionTemplate,
                                      val releaseRepository: ReleaseRepository,
                                      val jobRepository: JobRepository,
                                      val taskRepository: TaskRepository,
                                     ) extends Upgrade with Logging with BatchSupport with ParallelSupport with TransactionSupport {

  override def upgradeVersion(): Version = Version.valueOf(XL_RELEASE_COMPONENT, "10.2.0#1")

  override def doUpgrade(): Boolean = {
    logger.info("Migrating queued tasks to scheduled in progress jobs")

    val tasksIdsToProcess = getAllQueuedTaskIds
    logger.info(s"Found ${tasksIdsToProcess.size} tasks to process")

    doInBatch(tasksIdsToProcess.groupBy(Ids.releaseIdFrom), itemsName = "releases") { batch =>
      doInParallel(batch.items) { case (releaseId, taskIds) =>
        doInTransaction {
          Try(upgradeRelease(releaseId, taskIds.toSet))
            .recover(logger.error(s"Unable to process release $releaseId, giving up", _))
        }
      }
    }

    logger.info("Finished migrating queued tasks to scheduled in progress jobs")
    true
  }

  private def getAllQueuedTaskIds: mutable.Seq[String] = {
    jdbcTemplate.query(s"SELECT ${TASKS.TASK_ID} FROM ${TASKS.TABLE} WHERE ${TASKS.STATUS} = '${TaskStatus.QUEUED.value()}' OR ${TASKS.STATUS} = '${TaskStatus.ABORT_SCRIPT_QUEUED.value()}' OR ${TASKS.STATUS} = '${TaskStatus.FAILURE_HANDLER_QUEUED.value()}'",
      new RowMapper[String] {
        override def mapRow(rs: ResultSet, rowNum: Int): String = rs.getString(1)
      }).asScala
  }

  //noinspection ScalaStyle
  private def upgradeRelease(releaseId: String, taskIds: Set[String]): Unit = {
    val release = releaseRepository.findById(releaseId, ResolveOptions.WITHOUT_DECORATORS.withReferences)
    val tasksToUpdate = release.getAllTasks.asScala.filter(task => taskIds.contains(Ids.getFolderlessId(task.getId))).toSeq
    tasksToUpdate.foreach { task =>
      task.generateExecutionId()
      task.getStatus match {
        case TaskStatus.QUEUED if task.shouldFacetBeChecked() =>
          task.setStatus(TaskStatus.FACET_CHECK_IN_PROGRESS)
          jobRepository.create(JobRow(FacetCheckJob(new TaskSoftReference[Task](task, () => task))))
        case TaskStatus.QUEUED if task.shouldPreconditionBeChecked() =>
          task.setStatus(TaskStatus.PRECONDITION_IN_PROGRESS)
          jobRepository.create(JobRow(PreconditionJob(new TaskSoftReference[Task](task, () => task))))
        case TaskStatus.QUEUED =>
          task.setStatus(TaskStatus.IN_PROGRESS)
          jobRepository.create(JobRow(task match {
            case scriptTask: ResolvableScriptTask => ScriptTaskJob(new TaskSoftReference[ResolvableScriptTask](scriptTask, () => scriptTask))
            case customScriptTask: CustomScriptTask if customScriptTask.hasNextScriptToExecute => NextCustomScriptTaskJob(new TaskSoftReference[CustomScriptTask](customScriptTask, () => customScriptTask), Duration.Zero)
            case customScriptTask: CustomScriptTask => CustomScriptTaskJob(new TaskSoftReference[CustomScriptTask](customScriptTask, () => customScriptTask))
            case createReleaseTask: CreateReleaseTask => CreateReleaseTaskJob(new TaskSoftReference[CreateReleaseTask](createReleaseTask, () => createReleaseTask))
            case notificationTask: NotificationTask => NotificationTaskJob(new TaskSoftReference[NotificationTask](notificationTask, () => notificationTask))
          }))
        case TaskStatus.ABORT_SCRIPT_QUEUED if task.asInstanceOf[CustomScriptTask].hasNextScriptToExecute =>
          task.setStatus(TaskStatus.ABORT_SCRIPT_IN_PROGRESS)
          jobRepository.create(JobRow(NextCustomScriptTaskJob(new TaskSoftReference[CustomScriptTask](task.asInstanceOf[CustomScriptTask], () => task.asInstanceOf[CustomScriptTask]), Duration.Zero)))
        case TaskStatus.ABORT_SCRIPT_QUEUED =>
          task.setStatus(TaskStatus.ABORT_SCRIPT_IN_PROGRESS)
          jobRepository.create(JobRow(CustomScriptTaskJob(new TaskSoftReference[CustomScriptTask](task.asInstanceOf[CustomScriptTask], () => task.asInstanceOf[CustomScriptTask]))))
        case TaskStatus.FAILURE_HANDLER_QUEUED =>
          task.setStatus(TaskStatus.FAILURE_HANDLER_IN_PROGRESS)
          jobRepository.create(JobRow(FailureHandlerJob(new TaskSoftReference[Task](task, () => task))))
        case _ => ()
      }
    }
    releaseRepository.update(release, release)
    taskRepository.batchUpdateTaskProperties(tasksToUpdate.toSet)
  }

}
