package com.xebialabs.xlrelease.api.internal;

import java.io.IOException;
import java.io.InputStream;
import java.util.List;
import java.util.stream.Collectors;
import javax.servlet.http.HttpServletRequest;
import javax.ws.rs.*;
import javax.ws.rs.core.Context;
import javax.ws.rs.core.MediaType;
import org.apache.commons.fileupload.FileItemIterator;
import org.apache.commons.fileupload.FileItemStream;
import org.apache.commons.fileupload.FileUploadException;
import org.apache.commons.fileupload.servlet.ServletFileUpload;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;
import com.codahale.metrics.annotation.Timed;

import com.xebialabs.xlrelease.config.XlrConfig;
import com.xebialabs.xlrelease.export.TemplateImportContext;
import com.xebialabs.xlrelease.export.TemplateImporter;
import com.xebialabs.xlrelease.param.IdParam;
import com.xebialabs.xlrelease.security.PermissionChecker;
import com.xebialabs.xlrelease.security.XLReleasePermissions;
import com.xebialabs.xlrelease.service.UploadService;
import com.xebialabs.xlrelease.views.AttachmentView;
import com.xebialabs.xlrelease.views.ImportResult;

import static com.xebialabs.xlrelease.repository.Ids.isNullId;

/**
 * <p>
 * Handle file uploads used in various features of Digital.ai Release.
 * </p>
 * <p>
 * Upload URIs are not secured, because with IE the upload component falls back to an iframe, where we cannot pass authentication headers.
 * In order to limit the risk of exposing unsecured URIs, we make them temporary: before performing an upload, the client requests a
 * one-time token, that it must append to the URI it submits to.
 * </p>
 * Results must be TEXT_PLAIN because of IE9 iframe download handling, ZIP type makes IE believes it's a download of zip file.
 */
@Path("/upload")
@Consumes({MediaType.MULTIPART_FORM_DATA})
@Produces({MediaType.TEXT_HTML + "; charset=utf-8"})
@Controller
public class UploadResource {
    private static final String JSON_SUFFIX = ".json";

    private PermissionChecker permissionChecker;

    private TemplateImporter templateImporter;

    private UploadService uploadService;

    @Autowired
    public UploadResource(PermissionChecker permissionChecker, TemplateImporter templateImporter, UploadService uploadService) {
        this.permissionChecker = permissionChecker;
        this.templateImporter = templateImporter;
        this.uploadService = uploadService;
    }

    @SuppressWarnings("unused")
    public UploadResource() {
    }

    @POST
    @Timed
    @Path("templates/zip")
    public List<ImportResult> importTemplates(@Context HttpServletRequest request, @QueryParam("folderId") String folderId) throws IOException, FileUploadException {
        if (isNullId(folderId)) {
            permissionChecker.check(XLReleasePermissions.CREATE_TEMPLATE);
        } else {
            permissionChecker.check(XLReleasePermissions.EDIT_TEMPLATE, folderId);
        }

        ServletFileUpload upload = new ServletFileUpload();
        FileItemIterator fileItems = upload.getItemIterator(request);
        while (fileItems.hasNext()) {
            FileItemStream item = fileItems.next();
            if (item.isFormField()) {
                continue;
            }
            try (InputStream stream = item.openStream()) {
                boolean isJson = item.getName().toLowerCase().endsWith(JSON_SUFFIX);
                boolean isDsl = item.getName().toLowerCase().endsWith(".zip");
                return templateImporter.importTemplate(stream, new TemplateImportContext(folderId, isJson, false, isDsl, null));
            }
        }
        throw new IllegalArgumentException("Missing file");
    }

    @POST
    @Timed
    @Path("attachment/{ciId}")
    public List<AttachmentView> addAttachment(@PathParam("ciId") @IdParam String ciId, @Context HttpServletRequest request) throws IOException {
        permissionChecker.checkEditAttachment(ciId);

        if (!ServletFileUpload.isMultipartContent(request)) {
            throw new BadRequestException("Expected multipart content");
        }

        ServletFileUpload upload = new ServletFileUpload();
        upload.setSizeMax(XlrConfig.getInstance().server().upload().maxSizeBytes());
        try {
            FileItemIterator fileItems = upload.getItemIterator(request);
            List<AttachmentView> returnedViews = this.uploadService.addAttachment(ciId, fileItems).stream()
                    .map(AttachmentView::new).collect(Collectors.toList());
            if (returnedViews.isEmpty()) {
                throw new BadRequestException("Expected file upload");
            }
            return returnedViews;
        } catch (FileUploadException ex) {
            throw new BadRequestException(ex);
        }
    }

}
