package com.xebialabs.deployit.plumbing;

import java.io.IOException;
import javax.servlet.*;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpSession;
import org.springframework.http.HttpStatus;
import org.springframework.security.authentication.AnonymousAuthenticationToken;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.context.SecurityContextImpl;

import com.xebialabs.deployit.security.PermissionEnforcer;
import com.xebialabs.deployit.security.Permissions;

public class MaintenanceModeAccessControlFilter implements Filter {

    private final PermissionEnforcer permissionEnforcer;

    private final boolean isApiAccessRestricted;

    public MaintenanceModeAccessControlFilter(PermissionEnforcer permissionEnforcer, boolean isApiAccessRestricted) {
        this.permissionEnforcer = permissionEnforcer;
        this.isApiAccessRestricted = isApiAccessRestricted;
    }

    @Override
    public void doFilter(final ServletRequest request, final ServletResponse response, final FilterChain chain) throws IOException, ServletException {
        HttpServletRequest httpServletRequest = (HttpServletRequest) request;
        HttpSession session = httpServletRequest.getSession(false);
        Authentication currentUserAuthentication = Permissions.getAuthentication();

        boolean sessionExists = session != null;
        boolean isAnonymousUser = currentUserAuthentication instanceof AnonymousAuthenticationToken;
        boolean isSessionlessAuthenticatedUser = currentUserAuthentication != null && !isAnonymousUser && !sessionExists;

        // authenticated (api and static resource) calls with JSESSIONID, e.g. calls from GUI
        // Unsecured Resource Request calls without any authentication attached e.g. calls to register license, setting Themes
        if (sessionExists) {
            boolean isStaticResourceRequest = currentUserAuthentication == null;
            boolean isUnsecuredResourceRequest = isStaticResourceRequest && session.getAttribute("SPRING_SECURITY_CONTEXT") == null;
            if (isUnsecuredResourceRequest) {
                chain.doFilter(request, response);
                return;
            }
            if (isStaticResourceRequest) {
                SecurityContextImpl sci = (SecurityContextImpl) session.getAttribute("SPRING_SECURITY_CONTEXT");
                currentUserAuthentication = sci.getAuthentication();
            }
            isAllowedAccessInMaintenanceMode(request, response, chain, currentUserAuthentication);
            return;
        }

        // api calls from e.g. Postman. They don't have to have a JSESSIONID and thus won't happen inside a session, but they have to be authenticated
        if (isSessionlessAuthenticatedUser) {
            if (isApiAccessRestricted) {
                isAllowedAccessInMaintenanceMode(request, response, chain, currentUserAuthentication);
                return;
            } else {
                chain.doFilter(request, response);
                return;
            }
        }

        // login without having a valid previous session (it requires static resources to render properly)
        chain.doFilter(request, response);
    }

    private void isAllowedAccessInMaintenanceMode(final ServletRequest request, final ServletResponse response, final FilterChain chain, final Authentication currentUserAuthentication) throws IOException, ServletException {
        if (permissionEnforcer.isAdmin(currentUserAuthentication)) {
            chain.doFilter(request, response);
        } else {
            HttpServletResponse httpResponse = (HttpServletResponse) response;
            httpResponse.setStatus(HttpStatus.SERVICE_UNAVAILABLE.value());
        }
    }

}