package com.xebialabs.xlrelease.actors.utils

import akka.actor.ActorSystem
import com.xebialabs.xlplatform.cluster.ClusterMode.Full
import com.xebialabs.xlrelease.actors._
import com.xebialabs.xlrelease.config.XlrConfig
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Component

import scala.concurrent.Await
import scala.concurrent.duration._
import scala.language.postfixOps

@Component
class ReleaseActorLifecycleUtils @Autowired()(val actorSystemHolder: ActorSystemHolder,
                                              val xlrConfig: XlrConfig,
                                              val releaseActorService: ReleaseActorService) extends ActorLifecycleUtils {

  private lazy implicit val system: ActorSystem = actorSystemHolder.getInstance()

  def activateReleaseActorAndAwait(releaseId: String) = {
    releaseActorService.activate(releaseId)
    awaitReleaseActorStart(releaseId)
  }

  def terminateAllReleaseActorsAndAwait(terminationTimeout: FiniteDuration) = terminateAndAwait("Release*", terminationTimeout)

  def terminateReleaseActorAndAwait(releaseId: String, terminationTimeout: FiniteDuration) =
    terminateAndAwait(releaseId2ActorName(releaseId), terminationTimeout = terminationTimeout)

  private def terminateAndAwait(absoluteOrReleasePath: String, terminationTimeout: FiniteDuration) = {
    Await.result(findAndTerminate(toActorPath(absoluteOrReleasePath), terminationTimeout = terminationTimeout), 15 seconds)
  }

  private def awaitActorsStart(absoluteOrReleasePath: String) = {
    Await.result(actorStart(toActorPath(absoluteOrReleasePath)), 15 seconds)
  }

  private def awaitReleaseActorStart(releaseId: String) =
    awaitActorsStart(releaseId2ActorName(releaseId))

  private def toActorPath(path: String): String =
    if (path.startsWith("/"))
      path
    else if (xlrConfig.clusterMode == Full)
      "/system/sharding/release/*/" + path
    else
      "/user/releases/" + path

}
