package com.xebialabs.xlrelease.service

import com.xebialabs.xlrelease.domain.Task
import com.xebialabs.xlrelease.domain.distributed.events._
import com.xebialabs.xlrelease.events.{AsyncSubscribe, EventListener}
import com.xebialabs.xlrelease.repository.{InMemoryTaskSSERepository, TaskSSERepository}
import com.xebialabs.xlrelease.scheduler.logs.TaskLogCreated
import grizzled.slf4j.Logging
import org.jboss.resteasy.plugins.providers.sse.OutboundSseEventImpl
import org.springframework.stereotype.Service

import javax.ws.rs.core.MediaType
import javax.ws.rs.sse.SseEventSink
import scala.collection._
import scala.jdk.FutureConverters.CompletionStageOps


object TaskSSEService {
  private val LOG_EVENT = "TASK_LOG_EVENT"
  private val COMMENT_EVENT = "TASK_COMMENT_EVENT"
  private val ATTACHMENT_EVENT = "TASK_ATTACHMENT_EVENT"
}

@Service
@EventListener
class TaskSSEService extends Logging {

  private val sseRepository: TaskSSERepository = new InMemoryTaskSSERepository()

  private def newEventBuilder() = new OutboundSseEventImpl.BuilderImpl()

  def followTask(task: Task, sink: SseEventSink): Unit = {
    sseRepository.put(task.getId, sink)
  }

  @AsyncSubscribe
  def onEvent(event: DistributedXLReleaseEvent): Unit = {
    event match {
      case TaskLogCreated(taskId, executionId, _) =>
        sendEventToTask(taskId, TaskSSEService.LOG_EVENT, executionId)
      case e: DistributedCommentEvent =>
        sendEventToTask(e.taskId, TaskSSEService.COMMENT_EVENT, "")
      case e: DistributedAttachmentEvent =>
        sendEventToTask(e.containerId, TaskSSEService.ATTACHMENT_EVENT, "")
      case _ =>
      // nothing to do yet
    }
  }

  private def sendEventToTask(taskId: String, eventName: String, message: AnyRef): Unit = {
    val taskSSE = sseRepository.get(taskId)
    sendEvent(taskSSE, eventName, message)
  }

  private def sendEvent(sinks: Set[SseEventSink], eventName: String, message: AnyRef): Unit = {
    sinks.foreach(sink => {
      if (!sink.isClosed) {
        val event = newEventBuilder().name(eventName).mediaType(MediaType.APPLICATION_JSON_TYPE).data(message.getClass, message).build()
        sink.send(event).asScala.recover { e =>
          logger.warn("can't send SSE event", e)
          sink.close()
        }(scala.concurrent.ExecutionContext.parasitic)
      }
    })
  }
}
