package com.xebialabs.xlrelease.repository

import com.codahale.metrics.annotation.Timed
import com.xebialabs.deployit.booter.local.utils.Strings
import com.xebialabs.deployit.exception.NotFoundException
import com.xebialabs.xlrelease.domain.tasks.gate.TargetIdResolutionFailedException
import com.xebialabs.xlrelease.domain.{Dependency, PlanItem}
import com.xebialabs.xlrelease.repository.IdMatchers.{PhaseId, ReleaseId, TaskId}
import com.xebialabs.xlrelease.serialization.json.repository.ResolverRepository
import com.xebialabs.xlrelease.service.ArchivingService
import com.xebialabs.xlrelease.variable.VariableHelper
import com.xebialabs.xltype.serialization.CiReference
import grizzled.slf4j.Logging
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Component

import scala.util.{Failure, Try}


@Component
class DependencyTargetResolver @Autowired()(planItemRepository: PlanItemRepository,
                                            resolverRepository: ResolverRepository,
                                            archivingService: ArchivingService) extends Logging {

  @Timed
  def resolveTarget(dependency: Dependency): Unit = {
    val targetId = dependency.getTargetId
    if (Strings.isNotEmpty(targetId) &&
      !VariableHelper.containsVariables(targetId) &&
      !dependency.isArchived) {

      val dependencyTargetResolved = Try {
        // some scenarios:
        //  1. planItem can be null if resolve does not find _RELEASE_ if its archived before we pre-initialize proxy ref
        //     - what to do? simply don't set planItem (leave it as null) and try to resolve it from archived repo
        //  2. if target type cannot be deduced from the database - same - planItem will be null
        //  3. if target item cannot be loaded from the database while proxy is constrcuted (pre-initialized)
        //     - proxy is not going to be populated, but there is no exception - planItem is a proxy (just not pre-initialized)
        //  4. if there is any exception - this block will fail
        val targetProperty = dependency.getType.getDescriptor.getPropertyDescriptor("target")
        val ciRef = new CiReference(dependency, targetProperty, targetId)
        val planItem = resolverRepository.resolve(targetId, ciRef).asInstanceOf[PlanItem]
        //  execute planItemRepository.exists after we create a proxy so we can decide if we should try to resolve archived target
        if (planItemRepository.exists(targetId)) {
          dependency.setTarget(planItem)
          true
        } else {
          false
        }
      }.getOrElse(false)
      if (!dependencyTargetResolved) {
        Try(resolveArchivedTarget(targetId, dependency.getId)).map { target =>
          dependency.setTarget(target)
          dependency.archive()
        }.recoverWith {
          case _: NotFoundException =>
            Failure(new TargetIdResolutionFailedException(s"Could not find target on dependency ${dependency.getId} by ID $targetId"))
        }.get
      }
    }
  }

  @Timed
  def populateArchivedTargetId(dependency: Dependency): Unit = {
    val targetId = dependency.getTargetId
    val targetExists = Try(planItemRepository.exists(targetId)).getOrElse(false)
    if (Strings.isNotEmpty(targetId) && !VariableHelper.containsVariables(targetId) && !dependency.isArchived && !targetExists) {
      Try(resolveArchivedTarget(targetId, dependency.getId)).map { target =>
        dependency.setTarget(target)
        dependency.archive()
      }.recover {
        case e: Throwable =>
          logger.error(s"Unable to populate target on dependency ${dependency.getId} by ID $targetId", e)
      }
    }
  }

  private def resolveArchivedTarget(targetId: String, dependencyId: String): PlanItem = targetId match {
    case ReleaseId(_) => archivingService.getRelease(targetId)
    case PhaseId(_) => archivingService.getPhase(targetId)
    case TaskId(_) => archivingService.getTask(targetId)
    case _ => throw new TargetIdResolutionFailedException(s"Invalid archived target on dependency $dependencyId, " +
      s"it is not a plan item: $targetId")
  }
}

