package com.xebialabs.xlrelease.api.internal;

import java.text.Collator;
import java.util.*;
import java.util.concurrent.ExecutionException;
import java.util.stream.Collectors;
import javax.ws.rs.*;
import javax.ws.rs.core.MediaType;
import javax.ws.rs.core.Response;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;
import com.codahale.metrics.annotation.Timed;
import com.google.common.base.Function;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.google.common.collect.Maps;

import com.xebialabs.deployit.security.UserService;
import com.xebialabs.deployit.security.authentication.UserAlreadyExistsException;
import com.xebialabs.xlplatform.security.dto.PasswordValidationResult;
import com.xebialabs.xlrelease.api.v1.forms.UserAccount;
import com.xebialabs.xlrelease.config.XlrConfig;
import com.xebialabs.xlrelease.domain.UserProfile;
import com.xebialabs.xlrelease.domain.validators.UserAccountValidator;
import com.xebialabs.xlrelease.security.PermissionChecker;
import com.xebialabs.xlrelease.security.SessionService;
import com.xebialabs.xlrelease.service.UserProfileService;
import com.xebialabs.xlrelease.service.Users;
import com.xebialabs.xlrelease.user.User;
import com.xebialabs.xlrelease.views.UserView;

import static com.google.common.base.Preconditions.checkArgument;
import static com.xebialabs.deployit.booter.local.utils.Strings.isNotBlank;
import static com.xebialabs.deployit.security.permission.PlatformPermissions.EDIT_SECURITY;
import static com.xebialabs.xlrelease.security.XLReleasePermissions.isAdmin;
import static javax.ws.rs.core.Response.Status.NOT_FOUND;

/**
 * The user accounts (either internal or external) known to Digital.ai Release.
 */
@Path("/users")
@Consumes({MediaType.APPLICATION_JSON})
@Produces({MediaType.APPLICATION_JSON})
@Controller
public class UserAccountResource {
    private static final String ALL_USERS_KEY = "ALL_USERS";

    private UserService userService;
    private Users users;
    private UserProfileService userProfileService;
    private PermissionChecker permissionChecker;
    private SessionService sessionService;
    private UserAccountValidator userAccountValidator;
    private final XlrConfig xlrConfig;
    private LoadingCache<String, List<UserView>> usersCache;

    @Autowired
    public UserAccountResource(UserService userService, Users users, UserProfileService userProfileService,
                               PermissionChecker permissionChecker, SessionService sessionService, UserAccountValidator userAccountValidator) {
        this.userService = userService;
        this.users = users;
        this.userProfileService = userProfileService;
        this.permissionChecker = permissionChecker;
        this.sessionService = sessionService;
        this.userAccountValidator = userAccountValidator;
        this.xlrConfig = XlrConfig.getInstance();
        this.usersCache = CacheBuilder.newBuilder().maximumSize(xlrConfig.usersCacheMaxSize())
                .refreshAfterWrite(xlrConfig.usersCacheDuration(), xlrConfig.usersCacheDurationUnit())
                .build(new CacheLoader<>() {
                    @Override
                    public List<UserView> load(final String key) {
                        return getUserAccounts().stream().map(user -> new UserView(user.getUsername(), user.getFullName())).collect(Collectors.toList());
                    }
                });
    }

    @GET
    public List<UserAccount> find() {
        permissionChecker.check(EDIT_SECURITY);
        return getUserAccounts();
    }

    @GET
    @Timed
    @Path("names")
    public List<UserView> getUsernames() {
        // Unchecked security for auto-completion.
        try {
            return this.usersCache.get(ALL_USERS_KEY);
        } catch (ExecutionException e) {
            throw new RuntimeException(e);
        }
    }

    private List<UserAccount> getUserAccounts() {
        Map<String, UserProfile> profiles = Maps.uniqueIndex(userProfileService.findAll(), BY_NAME);
        List<String> internalNames = users.getRepositoryUsernames();

        SortedSet<String> allNames = new TreeSet<>(Collator.getInstance());
        allNames.addAll(profiles.keySet());

        List<UserAccount> accounts = new ArrayList<>(allNames.size());
        for (String name : allNames) {
            accounts.add(new UserAccount(name, profiles.get(name), isInternalUser(internalNames, name)));
        }
        return accounts;
    }

    private boolean isInternalUser(List<String> internalNames, String name) {
        for (String internalName : internalNames) {
            if (name.equalsIgnoreCase(internalName)) {
                return true;
            }
        }
        return false;
    }

    @POST
    public void create(UserAccount account) {
        permissionChecker.check(EDIT_SECURITY);
        userAccountValidator.check(account);
        checkNoUserProfileAlreadyExists(account.getUsername());
        boolean created = insertUser(account);
        if (!created) {
            throw new UserAlreadyExistsException(account.getUsername());
        } else {
            userProfileService.save(account.toUserProfile());
            this.usersCache.invalidateAll();
        }
    }

    @PUT
    @Timed
    public Response update(UserAccount account) {
        permissionChecker.check(EDIT_SECURITY);
        String username = account.getUsername();
        checkArgument(isNotBlank(username), "User name cannot be empty.");

        UserProfile profile = userProfileService.findByUsername(username);

        if (profile == null) {
            return Response.status(NOT_FOUND).build();
        }

        profile.setEmail(account.getEmail());
        profile.setFullName(account.getFullName());
        profile.setLoginAllowed(account.isLoginAllowed());
        userProfileService.updateProfile(profile);

        if (account.hasPassword() && users.userExistsInRepository(username)) {
            userAccountValidator.checkPassword(account.getPassword());
            if (isUsernameLoggedIn(username)) {
                userService.modifyPassword(username, account.getPassword(), account.getPreviousPassword());
            } else {
                userService.modifyPassword(username, account.getPassword());
            }
            sessionService.disconnect(username);
        }

        return Response.ok().build();
    }

    @DELETE
    @Timed
    public void delete(UserAccount account) {
        permissionChecker.check(EDIT_SECURITY);
        String username = account.getUsername();

        checkArgument(isNotBlank(username), "User name cannot be empty.");
        checkArgument(!isAdmin(username), "Admin user cannot be deleted.");

        userService.delete(username);
        userProfileService.deleteByUsername(username);
        sessionService.disconnect(username);
        this.usersCache.invalidateAll();
    }

    @POST
    @Timed
    @Path("validatePassword")
    public List<PasswordValidationResult> validatePassword(UserAccount account) {
        String password = account.getPassword() == null ? "" : account.getPassword();
        return userAccountValidator.validatePassword(password);
    }

    private boolean isUsernameLoggedIn(String username) {
        return username.equalsIgnoreCase(User.AUTHENTICATED_USER.getName());
    }

    /**
     * Checks for existing external users before creating internal user.
     *
     * @param username
     */
    private void checkNoUserProfileAlreadyExists(String username) {
        boolean userProfileExists = userProfileService.findByUsername(username) != null;
        if (userProfileExists) {
            throw new UserAlreadyExistsException(username);
        }
    }

    /**
     * @return false if the account already existed
     */
    private boolean insertUser(UserAccount account) {
        try {
            userService.create(account.getUsername(), account.getPassword());
        } catch (UserAlreadyExistsException e) {
            return false;
        }
        return true;
    }

    private static final Function<UserProfile, String> BY_NAME = new Function<UserProfile, String>() {
        @Override
        public String apply(UserProfile profile) {
            return extractName(profile.getId());
        }

        private String extractName(String id) {
            return id.replace(UserProfileService.ROOT + "/", "");
        }
    };

}
