package com.xebialabs.xlrelease.api.internal

import com.xebialabs.deployit.booter.local.utils.Strings.isNotBlank
import com.xebialabs.deployit.checks.Checks.checkArgument
import com.xebialabs.deployit.security.permission.PlatformPermissions.ADMIN
import com.xebialabs.xlrelease.actors.ReleaseActorService
import com.xebialabs.xlrelease.api.ApiService.{DEFAULT_RESULTS_PER_PAGE, PAGE, RESULTS_PER_PAGE}
import com.xebialabs.xlrelease.scheduler.filters.JobFilters
import com.xebialabs.xlrelease.scheduler.repository.JobRepository
import com.xebialabs.xlrelease.scheduler.{JobOverview, WorkManager}
import com.xebialabs.xlrelease.security.PermissionChecker
import com.xebialabs.xlrelease.utils.SortSupport
import com.xebialabs.xlrelease.views.CommentTasksView
import grizzled.slf4j.Logging
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.data.domain.{Page, PageRequest}
import org.springframework.stereotype.Controller

import javax.ws.rs._
import javax.ws.rs.core.{Context, MediaType, UriInfo}
import scala.jdk.CollectionConverters._
import scala.util.Try

@Path("/jobs")
@Consumes(Array(MediaType.APPLICATION_JSON))
@Produces(Array(MediaType.APPLICATION_JSON))
@Controller
class JobResource @Autowired()(permissions: PermissionChecker,
                               releaseActorService: ReleaseActorService,
                               workManager: WorkManager,
                               jobRepository: JobRepository) extends Logging {

  @GET
  @Path("/tasks")
  def searchTaskJobs(@BeanParam jobFilters: JobFilters,
                     @DefaultValue("0") @QueryParam(PAGE) page: Int,
                     @DefaultValue("100") @QueryParam(RESULTS_PER_PAGE) resultsPerPage: Int,
                     @Context uriInfo: UriInfo): Page[JobOverview] = {
    permissions.check(ADMIN)

    checkArgument(resultsPerPage <= DEFAULT_RESULTS_PER_PAGE, "Number of results per page cannot be more than %d", DEFAULT_RESULTS_PER_PAGE)
    val pageable = PageRequest.of(page, resultsPerPage, SortSupport.toSort(uriInfo))
    jobRepository.findAllJobOverview(jobFilters, pageable)
  }

  @POST
  @Path("/tasks/abort")
  def abortTaskJobs(commentTasksView: CommentTasksView): Unit = {
    permissions.check(ADMIN)

    val taskIds = commentTasksView.getTaskIds
    val commentText = commentTasksView.getCommentText

    checkArgument(isNotBlank(commentText), "Comment is mandatory when aborting jobs.")
    logger.debug(s"Got total ${taskIds.size()} jobs to abort")
    val abortedTaskIds = releaseActorService.abortTasks(taskIds, commentText)
    val notAbortedTaskIds = taskIds.asScala.diff(abortedTaskIds.asScala).toSeq

    if (notAbortedTaskIds.nonEmpty) {
      logger.error(s"Unable to abort following taskIds via task manager: ${notAbortedTaskIds.mkString("[", ", ", "]")}. Trying to fail it.")
      val failedTaskIds = releaseActorService.failTasks(notAbortedTaskIds, commentText)
      for (failedTaskId <- failedTaskIds) {
        Try(workManager.abortJobByTaskId(failedTaskId)).recover {
          case e => logger.error(s"Unable to fail task '$failedTaskId' via task manager", e)
        }
      }
      val notFailedTaskIds = notAbortedTaskIds.diff(failedTaskIds).mkString("'", ",", "'")
      logger.error(s"Unable to fail task ids: $notFailedTaskIds via task manager")
    }
  }

}
