package com.xebialabs.xlrelease.service

import com.codahale.metrics.annotation.Timed
import com.xebialabs.deployit.core.xml.WriteWithoutPasswordCiConverter
import com.xebialabs.deployit.exception.NotFoundException
import com.xebialabs.deployit.repository.{RepositoryAdapter, WorkDirContext}
import com.xebialabs.xlrelease.actors.ReleaseActorService
import com.xebialabs.xlrelease.api.v1.forms.{ReleaseOrderMode, ReleasesFilters}
import com.xebialabs.xlrelease.db.ArchivedReleases.AttachmentInfo
import com.xebialabs.xlrelease.db.{ArchivedReleases, ArchivedReleasesSearch}
import com.xebialabs.xlrelease.domain._
import com.xebialabs.xlrelease.domain.status.ReleaseStatus
import com.xebialabs.xlrelease.domain.utils.{AdaptiveReleaseId, FullReleaseId}
import com.xebialabs.xlrelease.domain.variables.Variable
import com.xebialabs.xlrelease.reports.filters.ReportFilter
import com.xebialabs.xlrelease.repository.Ids._
import com.xebialabs.xlrelease.repository._
import com.xebialabs.xlrelease.repository.query.ReleaseBasicDataExt
import com.xebialabs.xlrelease.serialization.json.utils.CiSerializerHelper.{newEncryptingConverter, serialize}
import grizzled.slf4j.Logging
import org.joda.time.DateTime
import org.joda.time.format.DateTimeFormat
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Service
import org.springframework.transaction.annotation.Isolation.READ_COMMITTED
import org.springframework.transaction.annotation.Propagation.REQUIRED
import org.springframework.transaction.annotation.Transactional

import java.util
import java.util.{Date, Locale, List => JList, Set => JSet}
import scala.jdk.CollectionConverters._
import scala.util.Try

object ArchivingService extends Logging {

  def getMonthYear(startDate: DateTime): String = {
    s"${DateTimeFormat.forPattern("MM").withLocale(Locale.ENGLISH).print(startDate)}.${startDate.getYear}"
  }
}

/**
 * The service responsible for storing releases into the relational database once they are completed.
 */
@Service
@Transactional(value = "reportingTransactionManager", propagation = REQUIRED, isolation = READ_COMMITTED, rollbackFor = Array(classOf[Throwable]))
class ArchivingService @Autowired()(val archivedReleases: ArchivedReleases,
                                    archivedReleasesSearch: ArchivedReleasesSearch,
                                    archivedActivityLogsSerializer: ArchivedActivityLogsSerializer,
                                    val repositoryAdapter: RepositoryAdapter,
                                    completedReleasesExportService: CompletedReleaseExportService,
                                    releaseRepository: ReleaseRepository,
                                    dependencyRepository: DependencyRepository,
                                    releaseActorService: ReleaseActorService,
                                    teamRepository: TeamRepository,
                                    securedCis: SecuredCis,
                                    releaseExtensionsRepository: ReleaseExtensionsRepository,
                                    commentService: CommentService,
                                    facetRepositoryDispatcher: FacetRepositoryDispatcher) extends ArchivedReleaseReader with Logging {

  facetRepositoryDispatcher.setupGenericFacetArchiveRepository(this)

  @Timed
  def archiveRelease(releaseId: String): Unit = {
    val release = repositoryAdapter.read[Release](releaseId)

    if (release == null) {
      logger.warn(s"Release '$releaseId' cannot be read from active database, not archiving.")
      if (!releaseRepository.exists(releaseId) && existsPreArchived(releaseId)) {
        logger.info(s"Setting Release '$releaseId' preArchived flag to false as it is no longer in the active database.")
        archivedReleases.setPreArchived(releaseId, preArchived = false)
      }
    } else if (!shouldArchive(release)) {
      logger.trace(s"Removing non archive-able release '$releaseId' from active database")
      deleteRelease(releaseId, release)
    } else if (ReleaseStatus.INACTIVE_STATUSES.contains(release.getStatus)) {
      if (existsPreArchived(releaseId)) {
        logger.trace(s"Archiving dependencies for '$releaseId'")

        // read fresh and fully decorated release
        releaseExtensionsRepository.decorate(release)
        release.getAllTasks.asScala.foreach(commentService.decorate)
        val updatedTeams = teamRepository.getTeams(securedCis.getEffectiveSecuredCi(release.getId))
        release.setTeams(updatedTeams)
        val facets = facetRepositoryDispatcher.liveRepository.genericRepository.findAllFacetsByRelease(release)
        release.getAllTasks.asScala.foreach { task =>
          task.setFacets(new util.ArrayList(facets.filter(_.getTargetId == task.getId).asJava))
        }

        archiveAttachments(release)
        archiveDependencies(release)

        logger.trace(s"Updating activity logs and releaseJson for '$releaseId'")
        val serializedRelease = serialize(release, new WriteWithoutPasswordCiConverter())
        val activityLogsJson = archivedActivityLogsSerializer.serializeActivityLogsOf(release.getId)
        if (!archivedReleases.update(releaseId, serializedRelease, activityLogsJson, preArchived = false)) {
          logger.warn(s"Could not update archived Release '$releaseId'.")
        } else {
          archivedReleases.updateViewers(releaseId, updatedTeams.asScala.toSeq)
        }
        // delete from active database (truly archived)
        deleteRelease(releaseId, release)
      } else {
        throw new IllegalStateException("Only pre-archived releases can be archived")
      }
    } else {
      logger.warn(s"Trying to pre-archive a release with invalid status ${release.getStatus}")
      throw new IllegalStateException("Only inactive releases can be archived")
    }
  }

  private def deleteRelease(releaseId: String, release: Release): Unit = if (releaseRepository.exists(releaseId)) {
    logger.trace(s"Deleting '$releaseId' from active database")
    releaseRepository.delete(releaseId)
    facetRepositoryDispatcher.liveRepository.findAllFacetsByRelease(release).foreach { facet =>
      facetRepositoryDispatcher.liveRepository.delete(facet.getId)
    }
  }

  private def archiveAttachments(release: Release): Unit = {
    logger.trace(s"Archiving attachments for '${release.getId}'")
    val previousWorkDirContext = Option(WorkDirContext.get())
    WorkDirContext.initWorkdir()
    try {
      release.getAttachments.asScala.foreach { attachment =>
        Try(attachment.getFile.getInputStream).foreach { data =>
          try {
            archivedReleases.insertAttachment(release.getId, AttachmentInfo(attachment.getId, attachment.getFile.getName, data))
          } finally {
            data.close()
          }
        }
      }
    } finally {
      Option(WorkDirContext.get).foreach(_.delete())
      WorkDirContext.clear()
      previousWorkDirContext.foreach(WorkDirContext.setWorkDir)
    }
  }

  private def archiveDependencies(release: Release): Unit = {
    val archivedDependencies = archiveAllOutgoingDependencies(release)
    dependencyRepository.archive(release, archivedDependencies)
    archiveAllIncomingDependencies(release.getId)
  }

  @Timed
  def deletePreArchiveRelease(releaseId: String): Boolean = archivedReleases.deleteReleaseFromArchive(releaseId)

  @Timed
  def preArchiveRelease(release: Release): Unit = {
    if (existsPreArchived(release.getId)) {
      logger.debug(s"$release is already pre-archived, nothing to do")
    } else {
      // decorate release with generic facets
      val facets = facetRepositoryDispatcher.liveRepository.genericRepository.findAllFacetsByRelease(release)
      release.getAllTasks.asScala.foreach { task =>
        task.setFacets(new util.ArrayList(facets.filter(_.getTargetId == task.getId).asJava))
      }

      release.setTeams(teamRepository.getTeams(securedCis.getEffectiveSecuredCi(release.getId)))
      val serializedRelease = serialize(release, new WriteWithoutPasswordCiConverter())

      if (shouldArchive(release)) {
        logger.debug(s"copying release $release to archive database, marked as pre-archived")

        archivedReleases.insert(release, serializedRelease, "", preArchived = true)

        // copy from live to archive
        val specialTypes = facetRepositoryDispatcher.supportedTypes
        facetRepositoryDispatcher.liveRepository.findAllFacetsByRelease(release)
          .filter(facet => specialTypes.exists(facet.getType.instanceOf))
          .foreach { liveFacet =>
            facetRepositoryDispatcher.archiveRepository.create(liveFacet)
          }

        releaseRepository.setPreArchived(release.getId, preArchived = true)
      }
      logger.debug(s"Running release ${release.getId} through export hooks")
      completedReleasesExportService.sendThroughExportHooks(release, serializedRelease)
    }
  }

  def shouldArchive(release: Release): Boolean = {
    !release.isTutorial && release.isArchiveRelease
  }

  def getRelease(releaseId: String): Release =
    getRelease(releaseId, includePreArchived = false)

  @Timed
  def getRelease(releaseId: String, includePreArchived: Boolean): Release = {
    getReleaseOption(releaseId, includePreArchived).getOrElse {
      throw new NotFoundException(s"Could not find archived release [$releaseId]")
    }
  }

  @Timed
  def getReleaseTitle(releaseId: String): String = {
    archivedReleases.getReleaseTitle(releaseId).getOrElse {
      throw new NotFoundException(s"Could not find archived release [$releaseId]")
    }
  }

  @Timed
  def getPhase(phaseId: String): Phase = {
    getReleaseItem[Phase](phaseId)(_.getPhase).getOrElse {
      throw new NotFoundException(s"Could not find archived release [${releaseIdFrom(phaseId)}]")
    }
  }

  @Timed
  def getTask(taskId: String): Task = {
    getReleaseItem[Task](taskId)(_.getTask).getOrElse {
      throw new NotFoundException(s"Could not find archived release [${releaseIdFrom(taskId)}]")
    }
  }

  @Timed
  def getVariable(variableId: String): Variable = {
    getVariableOption(variableId).getOrElse {
      throw new NotFoundException(s"Could not find archived release [${releaseIdFrom(variableId)}]")
    }
  }

  @Timed
  def exists(ciId: String): Boolean = {
    if (isReleaseId(ciId)) {
      archivedReleases.exists(ciId)
    } else if (isPhaseId(ciId)) {
      getReleaseItem[Phase](ciId)(_.getPhase).isDefined
    } else if (isTaskId(ciId)) {
      getReleaseItem[Task](ciId)(_.getTask).isDefined
    } else if (isVariableId(ciId)) {
      getVariableOption(ciId).isDefined
    } else {
      false
    }
  }

  @Timed
  def existsPreArchived(ciId: String): Boolean = {
    if (isReleaseId(ciId)) {
      archivedReleases.existsPreArchived(ciId)
    } else {
      false
    }
  }

  @Timed
  def existsByName(releaseName: String): Boolean = {
    archivedReleases.existsByName(releaseName)
  }

  @Timed
  def attachmentExists(attachmentId: String): Boolean = {
    archivedReleases.attachmentExists(attachmentId)
  }

  @Timed
  def getAttachment(attachmentId: String): Attachment = {
    getAttachmentOption(attachmentId).getOrElse {
      throw new NotFoundException(s"Could not find archived attachment [$attachmentId]")
    }
  }

  @Timed
  def getAttachmentOption(attachmentId: String): Option[Attachment] = {
    for {
      release <- getReleaseOption(releaseIdFrom(attachmentId))
      attachment <- Option(release.getAttachment(attachmentId))
      overthereFile <- archivedReleases.getAttachment(attachmentId)
    } yield {
      attachment.setFile(overthereFile)
      attachment
    }
  }

  @Timed
  def searchReleases(filters: ReleasesFilters): JList[Release] = {
    searchReleases(filters, None, None)
  }

  @Timed
  def searchReleases(filters: ReleasesFilters, limit: Long, offset: Long): JList[Release] = {
    searchReleases(filters, Option(limit), Option(offset))
  }

  @Timed
  def findShortReleaseIdsWithFolderNameAndOrderCriterion(filters: Seq[ReportFilter],
                                                         order: Option[ReleaseOrderMode],
                                                         limit: Option[Long] = None,
                                                         offset: Option[Long] = None
                                                        ): List[(AdaptiveReleaseId, Any)] = {
    archivedReleasesSearch.searchReleaseIdsAndOrderCriterion(filters, order, limit, offset)
  }

  @Timed
  def countReleasesByStatus(filters: ReleasesFilters): Map[ReleaseStatus, Int] = {
    archivedReleasesSearch.countReleasesByStatus(filters)
  }

  @Timed
  def countReleases(filters: Seq[ReportFilter]): Map[ReleaseStatus, Int] = {
    archivedReleasesSearch.countReleasesByStatus(filters)
  }

  @Timed
  def findArchivableReleaseIds(date: Date, pageSize: Integer): Seq[String] = {
    archivedReleases.findArchivableReleaseIds(date, pageSize)
  }

  @Timed
  def findPurgableReleaseIds(date: Date, pageSize: Integer): Seq[String] = {
    archivedReleases.findPurgableReleaseIds(date, pageSize)
  }

  @Timed
  def purgeArchivedRelease(releaseId: String): Unit = {
    archivedReleases.deleteReleaseFromArchive(releaseId)
  }

  @Timed
  def archiveAllIncomingDependencies(releaseId: String): Unit = {
    val incomingDependencies = dependencyRepository.findAllIncomingDependencies(Seq(releaseId), Seq.empty, referencingChildren = true)
    incomingDependencies
      .groupBy(d => normalizeId(releaseIdFrom(d.getId)))
      .foreach {
        case (referencingReleaseId, rawDependenciesToArchive) =>
          releaseActorService.archiveDependencies(referencingReleaseId, rawDependenciesToArchive.map(_.getId))
      }
  }

  @Timed
  def getAllTags(limitNumber: Int): JSet[String] = {
    archivedReleases.findAllTags(limitNumber).asJava
  }

  private def searchReleases(filters: ReleasesFilters, limit: Option[Long], offset: Option[Long]): JList[Release] = {
    val releaseJsons = archivedReleasesSearch.searchReleases(filters, limit, offset)
    releaseJsons.map(json => deserializeArchivedRelease(json, repositoryAdapter, newEncryptingConverter()))
  }.asJava

  def searchReleasesByReleaseIds(releaseIds: Seq[String]): Seq[Release] = {
    val releaseJsons = archivedReleasesSearch.searchReleasesByIds(releaseIds)
    releaseJsons.map(json => deserializeArchivedRelease(json, repositoryAdapter, newEncryptingConverter()))
  }

  def searchReleasesBasicExtByReleaseIds(releaseIds: Seq[String]): Seq[ReleaseBasicDataExt] =
    archivedReleasesSearch.searchReleasesBasicInfoByIds(releaseIds.map(FullReleaseId(_)
      .withOnlyOneParentOrApplicationsForArchiveDb()))

  private def getReleaseItem[T <: PlanItem](itemId: String)(getItem: Release => String => T): Option[T] =
    getReleaseOption(releaseIdFrom(itemId))
      .flatMap(r => Option(getItem(r)(itemId)))

  private def getVariableOption(variableId: String): Option[Variable] = {
    getReleaseOption(releaseIdFrom(variableId)).flatMap { release =>
      release.getVariables.asScala.find(_.getId == variableId)
    }
  }

  private def archiveAllOutgoingDependencies(release: Release): Seq[Dependency] = {
    val dependencies = release.getAllGates.asScala
      .flatMap(_.getDependencies.asScala)
      .filter(!_.isArchived)
    dependencies.foreach(_.archive())
    dependencies.toSeq
  }
}
