package com.xebialabs.xlrelease.script

import com.xebialabs.deployit.security.Permissions.getAuthentication
import com.xebialabs.xlrelease.domain.CustomScriptTask.WAIT_FOR_SIGNAL_PROPERTY_NAME
import com.xebialabs.xlrelease.domain.{CustomScriptTask, Task}
import com.xebialabs.xlrelease.script.DefaultScriptService.CustomScriptTaskResults
import com.xebialabs.xlrelease.script.ScriptServiceHelper.{addExceptionToExecutionLog, getAttachmentIdFromExecutionLog}
import com.xebialabs.xlrelease.utils.SensitiveValueScrubber

import java.io.Writer

trait GenericTaskScriptLogic {
  self: DefaultScriptService =>

  def executeGenericTaskScript(task: Task, scriptContext: XlrScriptContext): ReactiveTaskResult = {
    val executionId = task.getExecutionId
    val taskId = task.getId
    registerScriptExecution(taskId, executionId)
    registerWriterForTask(task, SensitiveValueScrubber.disabled)
    try {
      authenticationService.loginScriptUser(task)
      scriptContext.setWriter(executionLog)
      val statementResult = executeScriptWithLifecycle(scriptContext)
      onSuccess(task.asInstanceOf[CustomScriptTask], scriptContext)(executionLog, statementResult)
    } catch {
      case exception: Exception =>
        onFailure(task.asInstanceOf[CustomScriptTask])(executionLog, exception)
    } finally {
      closeWriter()
      authenticationService.logoutScriptUser()
      finishScript(executionId)
    }
  }

  private def onSuccess(task: CustomScriptTask, context: XlrScriptContext)(executionLog: Writer, resultStatement: AnyRef): ReactiveTaskResult = {
    val resultVariable = context.getAttribute(DefaultScriptService.RESULT_ATTRIBUTE)
    val results = new CustomScriptTaskResults(
      ScriptServiceHelper.extractTransitionalAndOutputPropertyValues(task, context, SensitiveValueScrubber.disabled()),
      task.getExecutionId, task.getStatusLine, task.getNextScriptPath, task.getInterval)

    (resultStatement, resultVariable) match {
      case (null, null) => buildFailure(task, "Condition script did not return anything", executionLog)
      case (_, resVar: java.lang.Boolean) => buildResult(resVar.asInstanceOf[Boolean], task, results, executionLog)
      case (stmt: java.lang.Boolean, _) => buildResult(stmt.asInstanceOf[Boolean], task, results, executionLog)
      case _ => buildFailure(task, "Condition script must return a boolean value", executionLog)
    }
  }

  private def onFailure(task: CustomScriptTask)(executionLog: Writer, exception: Exception): ReactiveTaskResult = {
    addExceptionToExecutionLog(exception, executionLog, "Unexpected exception when matching event on task '{}': ", task.getId)
    failed(task, executionLog)
  }

  private def buildResult(isMatched: Boolean, task: CustomScriptTask, scriptResult: CustomScriptTaskResults, executionLog: Writer): ReactiveTaskResult = {
    if (isMatched) {
      executionLog.append(DefaultScriptService.MATCHED_EVENT_MSG)
      scriptResult.getOutputVariables.put(WAIT_FOR_SIGNAL_PROPERTY_NAME, false)
      matched(task, scriptResult, executionLog)
    } else {
      unmatched(task, scriptResult, executionLog)
    }
  }

  private def matched(task: CustomScriptTask, scriptResult: CustomScriptTaskResults, executionLog: Writer): MatchedEventResult = {
    MatchedEventResult(task.getId, task.getExecutionId, getLog(executionLog), getAttachmentId(executionLog), scriptResult, getAuthentication)
  }

  private def unmatched(task: CustomScriptTask, scriptResult: CustomScriptTaskResults, executionLog: Writer): UnmatchedEventResult = {
    UnmatchedEventResult(task.getId, task.getExecutionId, getLog(executionLog), getAttachmentId(executionLog), scriptResult, getAuthentication)
  }

  private def buildFailure(task: CustomScriptTask, message: String, executionLog: Writer): FailedEventResult = {
    executionLog.append("Exception during execution:\n")
    executionLog.append(message)
    failed(task, executionLog)
  }

  private def failed(task: CustomScriptTask, executionLog: Writer): FailedEventResult = {
    FailedEventResult(task.getId, task.getExecutionId, getLog(executionLog), getAttachmentId(executionLog), getAuthentication)
  }

  private def getLog(executionLog: Writer): String = executionLog.toString

  private def getAttachmentId(executionLog: Writer): Option[String] = getAttachmentIdFromExecutionLog(executionLog)
}
