package com.xebialabs.xlrelease.metrics

import com.codahale.metrics.jmx.ObjectNameFactory
import com.xebialabs.xlrelease.metrics.MetricsConfiguration.METRICS_DOMAIN

import javax.management.{MalformedObjectNameException, ObjectName}

object XlrObjectNameFactory extends ObjectNameFactory {

  override def createName(`type`: String, originalDomain: String, originalName: String): ObjectName = {
    val (domain, name) = customDomainAndName(originalName)
    try {
      val objectName = new ObjectName(domain, "name", name)
      if (objectName.isPattern) {
        new ObjectName(domain, "name", ObjectName.quote(name))
      } else {
        objectName
      }
    } catch {
      case _: MalformedObjectNameException =>
        try {
          new ObjectName(domain, "name", ObjectName.quote(name))
        } catch {
          case e: MalformedObjectNameException =>
            throw new RuntimeException(e)
        }
    }
  }

  //noinspection ScalaStyle
  private[metrics] def customDomainAndName(name: String): (String, String) = {
    val xlrClass = """com\.xebialabs\.xlrelease\.(?:([\w\.]+)\.)?([A-Z]\w+(?:\.\w+)?)""".r
    val (domainSuffix, customName) = name match {
      case n if n.startsWith("connections") => ("handler", name)
      case n if n.startsWith("org.eclipse.jetty.webapp.WebAppContext") => ("handler", name.split('.').last)
      case n if n.startsWith("scheduler") => ("executors", name)
      case n if n.startsWith("releaseTrigger") => ("executors", name)
      case n if n.startsWith("riskCalculation") => ("executors", name)
      case n if n.startsWith("RepositoryPool") => ("pool.repository", name.stripPrefix("RepositoryPool.pool."))
      case n if n.startsWith("ReportingPool") => ("pool.reporting", name.stripPrefix("ReportingPool.pool."))
      case n if n.startsWith("ReleaseExecutionActor") => ("ReleaseExecutionActor.counts", name.stripPrefix("ReleaseExecutionActor:"))
      case xlrClass(suffix, className) if suffix != null && suffix.contains("service") => ("service", className)
      case xlrClass(suffix, className) if suffix != null && suffix.contains("repository") => ("repository", className)
      case xlrClass(suffix, className) => (suffix, className)
      case _ => (null, name)
    }
    val customDomain = if (domainSuffix == null) METRICS_DOMAIN else s"$METRICS_DOMAIN.$domainSuffix"
    (customDomain, customName)
  }
}
