package com.xebialabs.xlrelease.api.internal;

import javax.ws.rs.*;
import javax.ws.rs.core.MediaType;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;

import com.xebialabs.deployit.engine.spi.exception.DeployitException;
import com.xebialabs.deployit.engine.spi.exception.HttpResponseCodeResult;
import com.xebialabs.deployit.security.UserService;
import com.xebialabs.xlrelease.api.v1.forms.UserAccount;
import com.xebialabs.xlrelease.domain.UserProfile;
import com.xebialabs.xlrelease.domain.validators.UserAccountValidator;
import com.xebialabs.xlrelease.security.SessionService;
import com.xebialabs.xlrelease.service.UserProfileService;
import com.xebialabs.xlrelease.service.Users;

import static com.google.common.base.Preconditions.checkArgument;
import static com.xebialabs.deployit.booter.local.utils.Strings.isNotBlank;
import static com.xebialabs.deployit.security.Permissions.getAuthenticatedUserName;

/**
 * The user profile data associated with an Digital.ai Release account.
 */
@Path("/profile")
@Consumes({MediaType.APPLICATION_JSON})
@Produces({MediaType.APPLICATION_JSON})
@Controller
public class ProfileResource {
    private final UserProfileService userProfileService;

    private final UserService userService;

    private final Users users;

    private final SessionService sessionService;

    private final UserAccountValidator userAccountValidator;

    @Autowired
    public ProfileResource(UserProfileService userProfileService, UserService userService, Users users,
                           UserAccountValidator userAccountValidator, SessionService sessionService) {
        this.userProfileService = userProfileService;
        this.userService = userService;
        this.users = users;
        this.userAccountValidator = userAccountValidator;
        this.sessionService = sessionService;
    }

    @GET
    public UserAccount getProfile() {
        String authenticatedUsername = getAuthenticatedUserName();
        UserProfile userProfile = userProfileService.findByUsername(authenticatedUsername);

        return new UserAccount(userProfile.getName(), userProfile, users.userExistsInRepository(authenticatedUsername));
    }

    @PUT
    public UserAccount saveProfile(UserAccount account) {
        String authenticatedUsername = getAuthenticatedUserName();

        UserProfile profile = userProfileService.findByUsername(authenticatedUsername);
        profile.setEmail(account.getEmail());
        profile.setFullName(account.getFullName());

        profile.setDateFormat(account.getDateFormat());
        profile.setTimeFormat(account.getTimeFormat());
        profile.setFirstDayOfWeek(account.getFirstDayOfWeek());
        profile.setAnalyticsEnabled(account.isAnalyticsEnabled());
        profile.setTaskDrawerEnabled(account.isTaskDrawerEnabled());

        userProfileService.updateProfile(profile);

        if (account.hasPassword() && users.userExistsInRepository(authenticatedUsername)) {
            checkArgument(isNotBlank(account.getPreviousPassword()), "Previous password must be provided when changing the password.");
            userAccountValidator.checkPassword(account.getPassword());
            try {
                userService.modifyPassword(authenticatedUsername, account.getPassword(), account.getPreviousPassword());
                sessionService.disconnect(authenticatedUsername);
            } catch (IllegalArgumentException e) {
                throw new InvalidPreviousPasswordException(e);
            }
        }

        return new UserAccount(profile.getName(), profile, users.userExistsInRepository(authenticatedUsername));
    }

    @SuppressWarnings("serial")
    @HttpResponseCodeResult(statusCode = 400)
    public static class InvalidPreviousPasswordException extends DeployitException {
        public InvalidPreviousPasswordException(Throwable cause) {
            super(cause);
        }
    }
}
