package com.xebialabs.xlrelease.server.jetty;

import java.lang.management.ManagementFactory;
import java.util.concurrent.TimeUnit;
import org.eclipse.jetty.http.HttpCompliance;
import org.eclipse.jetty.jmx.MBeanContainer;
import org.eclipse.jetty.server.*;
import org.eclipse.jetty.util.ssl.SslContextFactory;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.boot.web.embedded.jetty.JettyServerCustomizer;
import com.codahale.metrics.Timer;
import com.codahale.metrics.jetty9.InstrumentedConnectionFactory;

import com.xebialabs.deployit.ServerConfiguration;
import com.xebialabs.xlrelease.config.XlrConfig;
import com.xebialabs.xlrelease.metrics.XlrMetricRegistry;

import static com.google.common.base.Strings.nullToEmpty;
import static com.xebialabs.deployit.ServerConfiguration.DEFAULT_IDLE_TIMEOUT;
import static com.xebialabs.deployit.ServerConfiguration.DEFAULT_STOP_TIMEOUT;

public class ReleaseJettyServerCustomizer {

    private static final Logger logger = LoggerFactory.getLogger(ReleaseJettyServerCustomizer.class);

    private ReleaseJettyServerCustomizer() {
        /**
         * Empty private constructor to prevent object creation
         */
    }

    public static JettyServerCustomizer create(final ServerConfiguration serverConfiguration, final XlrConfig xlrConfig) {
        return jettyServer -> {
            ServerConnector connector;

            if (serverConfiguration.isSsl()) {
                SslContextFactory.Server sslContextFactory = getSslContextFactory(serverConfiguration);
                connector = new ServerConnector(jettyServer, sslContextFactory, getHttpConnectionFactory(xlrConfig));
            } else {
                connector = new ServerConnector(jettyServer, getHttpConnectionFactory(xlrConfig));
            }

            connector.setIdleTimeout(DEFAULT_IDLE_TIMEOUT);
            connector.setHost(serverConfiguration.getHttpBindAddress());
            connector.setPort(serverConfiguration.getHttpPort());
            jettyServer.setConnectors(new Connector[] {connector});

            logger.info("Server listens on {}:{} ({})", connector.getHost(), connector.getPort(), serverConfiguration.isSsl() ? "secure" : "not secure");

            jettyServer.setStopTimeout(DEFAULT_STOP_TIMEOUT);

            MBeanContainer mbContainer = new MBeanContainer(ManagementFactory.getPlatformMBeanServer());
            jettyServer.addEventListener(mbContainer);
            jettyServer.addBean(mbContainer);
        };
    }

    private static SslContextFactory.Server getSslContextFactory(ServerConfiguration serverConfiguration) {
        logger.debug("Setting up Jetty to use SSL");
        SslContextFactory.Server sslContextFactory = new SslContextFactory.Server();
        sslContextFactory.setKeyStorePath(serverConfiguration.getKeyStorePath());
        sslContextFactory.setKeyStorePassword(serverConfiguration.getKeyStorePassword());
        sslContextFactory.setKeyManagerPassword(serverConfiguration.getKeyStoreKeyPassword());
        String protocol = serverConfiguration.getSslProtocol();
        if (!nullToEmpty(protocol).trim().isEmpty()) {
            sslContextFactory.setProtocol(protocol);
        }

        if (serverConfiguration.isMutualSsl()) {
            logger.debug("Setting up Jetty to use mutual SSL");
            sslContextFactory.setNeedClientAuth(true);
            sslContextFactory.setTrustStorePath(serverConfiguration.getTrustStorePath());
            sslContextFactory.setTrustStorePassword(serverConfiguration.getTrustStorePassword());
        }
        return sslContextFactory;
    }

    private static ConnectionFactory getHttpConnectionFactory(XlrConfig xlrConfig) {
        HttpConfiguration httpConfiguration = new HttpConfiguration();
        httpConfiguration.addCustomizer(new SecureRequestCustomizer(
                true,
                TimeUnit.SECONDS.convert(365, TimeUnit.DAYS),
                true
        ));
        httpConfiguration.setSendServerVersion(false);
        httpConfiguration.setSendXPoweredBy(false);

        ConnectionFactory connectionFactory = new HttpConnectionFactory(httpConfiguration, HttpCompliance.RFC2616);
        if (xlrConfig.metrics().enabled()) {
            Timer timer = XlrMetricRegistry.metricRegistry().timer("connections");
            connectionFactory = new InstrumentedConnectionFactory(connectionFactory, timer);
        }
        return connectionFactory;
    }
}
