package com.xebialabs.xlrelease.scheduler


import com.fasterxml.jackson.annotation.JsonIgnoreProperties
import com.xebialabs.xlrelease.domain._
import com.xebialabs.xlrelease.repository.Ids
import com.xebialabs.xlrelease.scheduler.JobStatus._
import com.xebialabs.xlrelease.scheduler.JobType._
import com.xebialabs.xlrelease.script.TaskSoftReference
import grizzled.slf4j.Logging

import java.time.Instant
import java.time.temporal.ChronoUnit
import java.util.concurrent.{Delayed, TimeUnit}
import scala.beans.BeanProperty
import scala.concurrent.duration.Duration
import scala.jdk.CollectionConverters._

@JsonIgnoreProperties(ignoreUnknown = true)
case class JobRow(@BeanProperty id: Long,
                  @BeanProperty node: String,
                  @BeanProperty jobType: JobType,
                  @BeanProperty status: JobStatus,
                  @BeanProperty version: Long,
                  @BeanProperty submitTime: Instant,
                  @BeanProperty reservationTime: Instant,
                  @BeanProperty startTime: Instant,
                  @BeanProperty scheduledStartTime: Instant,
                  @BeanProperty taskId: String,
                  @BeanProperty releaseUid: Integer,
                  @BeanProperty executionId: String,
                  @BeanProperty runnerId: String) {
  def this() = this(-1, null, null, null, -1, null, null, null, null, null, -1, null, null)
}

object JobRow {
  def apply(job: Job): JobRow = JobRow(
    job.id,
    job.node,
    job.jobType,
    job.status,
    job.version,
    job.submitTime,
    job.reservationTime,
    job.startTime,
    job.scheduledStartTime,
    job match {
      case taskJob: TaskJob[_] => taskJob.taskId
      case _ => null
    },
    job match {
      case taskJob: TaskJob[_] => taskJob.releaseUid
      case _ => null
    },
    job match {
      case taskJob: TaskJob[_] => taskJob.taskRef.getExecutionId
      case _ => null
    },
    job.runnerId
  )
}

sealed trait Job extends Delayed with Logging {
  // Should be true if job should be broadcasted to other nodes in the system
  def broadcast: Boolean

  @BeanProperty
  var id: Long = _

  var node: String = _

  var runnerId: String = _

  def jobType: JobType

  var status: JobStatus = QUEUED

  var version: Long = 0

  var submitTime: Instant = Instant.now().truncatedTo(ChronoUnit.MILLIS)

  var reservationTime: Instant = _

  val scheduledStartTime: Instant = Instant.now().truncatedTo(ChronoUnit.MILLIS)

  var startTime: Instant = _

  def capabilities(): Set[String] = Set("local")

  override def getDelay(unit: TimeUnit): Long = {
    val now = Instant.now()
    if (scheduledStartTime == null || scheduledStartTime.isBefore(now)) {
      0
    } else {
      java.time.Duration.between(now, scheduledStartTime).dividedBy(java.time.Duration.of(1, unit.toChronoUnit))
    }
  }

  override def compareTo(o: Delayed): Int = o match {
    case otherJob: Job =>
      if (scheduledStartTime == null) {
        if (otherJob.scheduledStartTime == null) {
          0
        } else {
          -1
        }
      } else {
        if (otherJob.scheduledStartTime == null) {
          1
        } else if (scheduledStartTime.isBefore(otherJob.scheduledStartTime)) {
          -1
        } else if (scheduledStartTime.isAfter(otherJob.scheduledStartTime)) {
          1
        } else {
          0
        }
      }
    case _ => throw new IllegalArgumentException("Queue should only contain Jobs")
  }
}


sealed trait TaskJob[U <: Task] extends Job {

  val broadcast: Boolean = true

  def taskRef: TaskSoftReference[_ <: U]

  /**
   * IMPORTANT: this is folder-less task Id, that starts with Release[a-f0-9]+
   */
  def taskId: String = if (null != taskRef) Ids.getFolderlessId(taskRef.getTaskId) else null

  def releaseUid: Integer = if (null != taskRef) taskRef.getReleaseCiUid else null
}

case class ScriptTaskJob(taskRef: TaskSoftReference[_ <: ResolvableScriptTask]) extends TaskJob[ResolvableScriptTask] {
  override val jobType = SCRIPT_TASK
}

object NextCustomScriptTaskJob {
  def apply(taskRef: TaskSoftReference[CustomScriptTask], delay: Duration): NextCustomScriptTaskJob =
    NextCustomScriptTaskJob(taskRef, Instant.now().truncatedTo(ChronoUnit.MILLIS).plusMillis(delay.toMillis))
}

case class CustomScriptTaskJob(taskRef: TaskSoftReference[CustomScriptTask]) extends TaskJob[CustomScriptTask] {
  override val jobType = CUSTOM_SCRIPT_TASK
}

case class NextCustomScriptTaskJob(taskRef: TaskSoftReference[CustomScriptTask], override val scheduledStartTime: Instant) extends TaskJob[CustomScriptTask] {
  override val jobType = NEXT_CUSTOM_SCRIPT_TASK
}

case class ContainerTaskJob(taskRef: TaskSoftReference[ContainerTask]) extends TaskJob[ContainerTask] {
  override val jobType = CONTAINER_TASK

  override def capabilities(): Set[String] = {
    // TODO taskRef - we might not want to load whole task just for it's capabilities
    val task = taskRef.get()
    task.getRequiredCapabilities.asScala.toSet
  }
}

case class CreateReleaseTaskJob(taskRef: TaskSoftReference[CreateReleaseTask]) extends TaskJob[CreateReleaseTask] {
  override val jobType = CREATE_RELEASE_TASK
}

case class NotificationTaskJob(taskRef: TaskSoftReference[NotificationTask]) extends TaskJob[NotificationTask] {
  override val jobType = NOTIFICATION_TASK
}

case class PreconditionJob(taskRef: TaskSoftReference[Task]) extends TaskJob[Task] {
  override val jobType = PRECONDITION
}

case class FacetCheckJob(taskRef: TaskSoftReference[Task]) extends TaskJob[Task] {
  override val jobType = FACET_CHECK
}

case class FailureHandlerJob(taskRef: TaskSoftReference[Task]) extends TaskJob[Task] {
  override val jobType = FAILURE_HANDLER
}

case class StopWorkerThread() extends Job {
  val broadcast: Boolean = false // should not be broadcasted to other nodes as it stops a thread
  override def jobType: JobType = STOP_WORKER_THREAD
}

case class FailJob(job: Job, errorMessage: Option[String]) extends Job {
  val broadcast: Boolean = true

  setId(job.id)
  override def jobType: JobType = FAIL_JOB
}
