package com.xebialabs.xlrelease.actors.kryoserializers

import com.esotericsoftware.kryo.io.{Input, Output}
import com.esotericsoftware.kryo.{Kryo, Serializer}
import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.deployit.plugin.api.udm.base.BaseConfigurationItem
import com.xebialabs.xltype.serialization.CiReference
import grizzled.slf4j.Logging

class CiReferenceSerializer extends Serializer[CiReference] with Logging {
  override def write(kryo: Kryo, output: Output, item: CiReference): Unit = {
    logger.debug(s"Serializing CiReference $item")
    kryo.writeObject(output, item.getCi.getType.toString)
    kryo.writeObject(output, item.getCi.getId)
    kryo.writeClassAndObject(output, item.getCi)
    val propertyDescriptorName = item.getProperty.getName
    kryo.writeObject(output, propertyDescriptorName)
    kryo.writeObject(output, item.getIds)
  }

  override def read(kryo: Kryo, input: Input, objectType: Class[CiReference]): CiReference = {
    val ciType = kryo.readObject(input, classOf[String])
    val ciId = kryo.readObjectOrNull(input, classOf[String])
    val ci = kryo.readClassAndObject(input).asInstanceOf[BaseConfigurationItem]
    // we need to set correct type on ci because it's not correctly populated while we're setting properites on the ci
    // (probably because of the order of fields in kryo deserialization)
    // in theory we should 1st correctly set Type of the CI and only then deserialize it's properties
    // but, at the moment we don't have ConfigurationItem specific Kryo serializer
    ci.setType(Type.valueOf(ciType))
    ci.setId(ciId)
    val propertyDescriptorName = kryo.readObject(input, classOf[String])
    val ids = kryo.readObject(input, classOf[java.util.ArrayList[String]])
    val propertyDescriptor = ci.getType.getDescriptor.getPropertyDescriptor(propertyDescriptorName)
    if (null == propertyDescriptor) throw new IllegalStateException(s"Property descriptor should NEVER be null for ci '$ciId' that references '$ids")
    val ciRef = new CiReference(ci, propertyDescriptor, ids)
    logger.debug(s"Deserialized CiReference $ciRef")
    ciRef
  }

}
