package com.xebialabs.xlrelease.service

import com.codahale.metrics.annotation.Timed
import com.xebialabs.deployit.checks.Checks.{checkArgument, checkNotNull}
import com.xebialabs.deployit.exception.NotFoundException
import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.deployit.repository.ItemInUseException
import com.xebialabs.deployit.security.Permissions.getAuthenticatedUserName
import com.xebialabs.xlrelease.api.v1.forms.ReleasesFilters
import com.xebialabs.xlrelease.configuration.FeatureSettings
import com.xebialabs.xlrelease.domain.Team.{FOLDER_OWNER_TEAMNAME, RELEASE_ADMIN_TEAMNAME, TEMPLATE_OWNER_TEAMNAME}
import com.xebialabs.xlrelease.domain.events._
import com.xebialabs.xlrelease.domain.folder.Folder
import com.xebialabs.xlrelease.domain.status.ReleaseStatus.{ACTIVE_STATUSES, PLANNED}
import com.xebialabs.xlrelease.domain.{BaseConfiguration, Release, Team}
import com.xebialabs.xlrelease.events.XLReleaseEventBus
import com.xebialabs.xlrelease.repository.Ids.{ROOT_FOLDER_ID, getParentId}
import com.xebialabs.xlrelease.repository.ReleaseSearchByParams.byAncestor
import com.xebialabs.xlrelease.repository._
import com.xebialabs.xlrelease.search.ReleaseSearchResult
import com.xebialabs.xlrelease.security.XLReleasePermissions.{EDIT_FOLDER_TEAMS, VIEW_FOLDER, VIEW_TRIGGER}
import com.xebialabs.xlrelease.security.{PermissionChecker, XLReleasePermissions}
import com.xebialabs.xlrelease.utils.CiHelper
import com.xebialabs.xlrelease.views.TemplateFilters
import grizzled.slf4j.Logging

import java.util.{List => JList}
import scala.collection.mutable
import scala.jdk.CollectionConverters._

class FolderService(folders: FolderRepository,
                    teamService: TeamService,
                    releaseRepository: ReleaseRepository,
                    taskService: TaskService,
                    configurationRepository: ConfigurationRepository,
                    ciIdService: CiIdService,
                    releaseSearchService: ReleaseSearchService,
                    permissions: PermissionChecker,
                    eventBus: XLReleaseEventBus,
                    archivingService: ArchivingService) extends Logging {

  val SLASH = "/"
  val SLASH_FOLDER = "/Folder"
  val TYPE_LIMITS = "xlrelease.Limits"
  val MAX_FOLDER_DEPTH = "maxFolderDepth"


  @Timed
  def checkFolderExists(folderId: String): Unit = folders.checkFolderExists(folderId)

  @Timed
  def exists(folderId: String): Boolean = folders.exists(folderId)

  @Timed
  def getTitle(folderId: String): String = {
    if (folderId == ROOT_FOLDER_ID) {
      throw new NotFoundException(s"Folder $ROOT_FOLDER_ID cannot be found")
    } else {
      folders.getTitle(folderId).getOrElse {
        throw new NotFoundException(s"Could not find Folder $folderId")
      }
    }
  }

  @Timed
  def findById(folderId: String, depth: Integer = Integer.MAX_VALUE): Folder = {
    folders.findById(folderId, depth.intValue()).getOrElse {
      throw new NotFoundException(s"Could not find Folder $folderId")
    }
  }

  @Timed
  def findViewableFoldersById(folderId: String, depth: Integer = Integer.MAX_VALUE, enforcePermission: Boolean = true): Folder = {
    folders.findViewableFoldersById(folderId, depth.intValue(), enforcePermission).getOrElse {
      throw new NotFoundException(s"Could not find Folder $folderId")
    }
  }

  @Timed
  def findByPath(path: String, depth: Int = Int.MaxValue): Folder = {
    folders.findByPath(path, depth)
  }

  @Timed
  def listViewableFolders(parentId: String,
                          page: Page,
                          decorateWithPermissions: Boolean = true,
                          enforcePermission: Boolean = true): JList[Folder] = {
    if (enforcePermission) {
      permissions.checkViewFolder(parentId)
    }
    folders.listViewableFolders(parentId, page, decorateWithPermissions, enforcePermission).asJava
  }

  @Timed
  def move(folderId: String, newParentId: String): Folder = {
    checkArgument(ROOT_FOLDER_ID != folderId, "Cannot move root folder")
    checkFolderExists(newParentId)
    validateFoldersDepthForMoving(folderId, newParentId)
    checkFolderCanBeMoved(folderId, Some(newParentId), "move")
    eventBus.publishAndFailOnError(FolderMovingAction(folderId, getParentId(folderId), newParentId))
    val folder = folders.move(folderId, newParentId)
    eventBus.publish(FolderMovedEvent(folder, getParentId(folderId), newParentId))
    folder
  }

  @Timed
  def rename(folderId: String, newName: String): Folder = {
    checkArgument(ROOT_FOLDER_ID != folderId, "Cannot rename root folder")
    checkArgument(!newName.trim.isEmpty, "Folder name cannot be blank")
    val folder = folders.rename(folderId, newName)
    eventBus.publish(FolderRenamedEvent(folder, newName))
    folder
  }

  @Timed
  def searchTemplates(folderId: String, page: Page, enforcePermission: Boolean = true): JList[Release] = {
    checkNotNull(folderId, "missing folderId")
    checkFolderExists(folderId)

    val filters = new TemplateFilters()
    filters.setParentId(folderId)

    releaseSearchService.searchTemplates(filters, page.page, page.resultsPerPage, page.depth.intValue(), enforcePermission).getReleases
  }

  @Timed
  def searchReleases(folderId: String, filters: ReleasesFilters, page: Page = Page.default): ReleaseSearchResult = {
    checkNotNull(folderId, "missing folderId")
    checkFolderExists(folderId)

    filters.setParentId(folderId)

    releaseSearchService.search(filters, page.page, page.resultsPerPage, page.depth.intValue())
  }

  @Timed
  def moveTemplate(folderId: String, templateId: String, shouldMergeTeams: Boolean = true): String = {
    val newTemplateId = s"$folderId/${Ids.getName(templateId)}"
    checkTemplateCanBeMoved(templateId, folderId)

    if (getParentId(templateId) == folderId) {
      logger.info(s"Attempted to move template $templateId to same folder $folderId. Will be ignored.")
      templateId
    } else {
      val templateTeams = teamService.getStoredTeams(templateId).asScala.toSeq
      eventBus.publishAndFailOnError(TemplateMovingAction(templateId, folderId))
      releaseRepository.move(templateId, newTemplateId)

      if (shouldMergeTeams) {
        mergeTeams(folderId, newTemplateId, templateTeams)
      } else {
        replaceTeams(folderId, newTemplateId)
      }

      eventBus.publish(ReleaseMovedEvent(templateId, newTemplateId))

      newTemplateId
    }
  }

  @Timed
  def create(parentId: String, folder: Folder): Folder = {
    val checkFolderDepth = parentId.split(SLASH_FOLDER)

    val allowedFolderDepth: Int = getAllowedFolderDepth()
    if (checkFolderDepth.length - 1 > allowedFolderDepth) {
      throw new IllegalArgumentException(s"Creating Folder exceeds allowed sub-folder depth ${allowedFolderDepth}")
    }
    create(parentId, folder, createDefaultTeams = true)
  }

  @Timed
  def create(parentId: String, folder: Folder, createDefaultTeams: Boolean): Folder = {
    val result = createWithoutPublishing(parentId, folder, createDefaultTeams)
    result.events.foreach(event => eventBus.publish(event))
    result.folder
  }

  private def validateFoldersDepthForMoving(folderId : String , newParentId: String) = {

    val allowedFolderDepth: Int = getAllowedFolderDepth()
    val newParentIdFolderDepth = newParentId.split(SLASH_FOLDER).length - 1

    if (newParentIdFolderDepth > allowedFolderDepth) {
      throw new IllegalArgumentException(s"Moving folder/folders is not within the allowed sub-folder depth ${allowedFolderDepth}")
    }

    val newFolderIdSplit = folderId.split(SLASH)
    val newFolderIdSplitLength = newFolderIdSplit.length
    val newFolderIdInDb = newFolderIdSplit(newFolderIdSplitLength - 1)

    val findChildrenOfNewFolder = folders.findNumberOfChildrenForAFolder(newFolderIdInDb)
    val newToBeFolderLength = newParentIdFolderDepth + findChildrenOfNewFolder

    if (newToBeFolderLength > allowedFolderDepth) {
      throw new IllegalArgumentException(s"Moving folder/folders is not within the allowed sub-folder depth ${allowedFolderDepth}")
    }
  }

  private def getAllowedFolderDepth(): Int = {
    val allLimitsProperty = configurationRepository.findFirstByType(Type.valueOf(TYPE_LIMITS)).orElse({
      val feature = Type.valueOf(TYPE_LIMITS).getDescriptor.newInstance[FeatureSettings]("")
      feature.generateId()
      feature
    })
    allLimitsProperty.getProperty(MAX_FOLDER_DEPTH)
  }

  case class FolderAndEvents(folder: Folder, events: Seq[XLReleaseEvent] = Seq.empty)

  def createWithoutPublishing(parentId: String, folder: Folder, createDefaultTeams: Boolean): FolderAndEvents = {
    checkArgument(!folder.getTitle.trim.isEmpty, "Folder name cannot be blank")
    checkArgument(folder.getTitle.length < 256, "Folder name must be 255 characters or less")
    checkFolderExists(parentId)
    folders.checkNameIsUnique(parentId, folder.getTitle)

    if (Ids.isNullId(folder.getId) || !Ids.isFolderId(folder.getId)) {
      folder.setId(ciIdService.getUniqueId(Type.valueOf(classOf[Folder]), parentId))
    } else {
      folder.setId(s"$parentId/${Ids.getName(folder.getId)}")
    }

    logger.info(s"Creating folder ${folder.getTitle} with id ${folder.getId}")

    val saved = folders.create(getParentId(folder.getId), folder)
    var events: Seq[XLReleaseEvent] = Seq(FolderCreatedEvent(folder))

    if (createDefaultTeams && parentId == ROOT_FOLDER_ID) {
      events = events ++ createDefaultTeamsForTheFolder(saved, parentId)
    } else {
      events = events :+ TeamsUpdatedEvent(folder.getId)
    }

    FolderAndEvents(saved, events)
  }

  @Timed
  def delete(folderId: String): Unit = {
    checkArgument(ROOT_FOLDER_ID != folderId, "Cannot delete root folder")
    checkFolderExists(folderId)
    checkFolderCanBeMoved(folderId, None, "delete")
    eventBus.publishAndFailOnError(FolderDeletingAction(folderId))
    logger.info(s"Deleting folder $folderId")
    val folder = folders.findById(folderId).get
    folders.delete(folderId, archiveOrDelete)
    eventBus.publish(FolderDeletedEvent(folder))
  }

  private def archiveOrDelete(folderUid: Int, releaseId: String): Unit = {
    val status = releaseRepository.getStatus(releaseId)
    logger.trace(s"archiveOrDelete($folderUid, $releaseId): status=$status [inactive? ${status.isInactive}]")
    if (status.isInactive) {
      val release = releaseRepository.findById(releaseId)
      if (archivingService.shouldArchive(release)) {
        archivingService.preArchiveRelease(release)
      }
      logger.trace(s"archiveRelease($releaseId)")
      archivingService.archiveRelease(releaseId)
    } else {
      logger.trace(s"deleteRelease($releaseId)")
      releaseRepository.delete(releaseId)
    }
  }

  private def mergeTeams(folderId: String, templateId: String, templateTeams: Seq[Team]): Unit = {
    if (templateTeams.nonEmpty) {
      val folderTeams = teamService.getEffectiveTeams(folderId).asScala.toSeq
      val folderTeamsContainerId = folderTeams.headOption.map(team => getParentId(team.getId)).getOrElse(folderId)
      val mergedTeams = mergeTemplateAndFolderTeams(folderTeams, templateTeams)

      val folderTitle = getTitle(folderId)

      teamService.saveTeamsToPlatform(folderTeamsContainerId, mergedTeams.asJava)
      teamService.deleteTeamsFromPlatform(templateId)

      eventBus.publish(TeamsMergedEvent(templateId, folderTitle))
    }
  }

  private def replaceTeams(folderId: String, templateId: String): Unit = {
    resetTaskTeam(templateId)
    val folderTitle = getTitle(folderId)
    eventBus.publish(TeamsRemovedInTemplateEvent(templateId, folderTitle))

    teamService.deleteTeamsFromPlatform(templateId)
  }

  private def mergeTemplateAndFolderTeams(folderTeams: Seq[Team], templateTeams: Seq[Team]): Seq[Team] = {
    val folderTeamsMap = collection.mutable.Map(folderTeams.map { team => (team.getTeamName, team) }.toMap.toSeq: _*)
    //loop through the templates and either add the new teams to the folder or update the folder
    templateTeams.foreach(templateTeam => {
      folderTeamsMap.get(templateTeam.getTeamName) match {
        case None =>
          templateTeam.setId(null)
          folderTeamsMap += templateTeam.getTeamName -> templateTeam

        case Some(folderTeam) =>
          val memberSet = folderTeam.getMembers.asScala.toSet ++ templateTeam.getMembers.asScala.toSet
          val roleSet = folderTeam.getRoles.asScala.toSet ++ templateTeam.getRoles.asScala.toSet
          val permissionSet = folderTeam.getPermissions.asScala.toSet ++ templateTeam.getPermissions.asScala.toSet
          folderTeam.setMembers(memberSet.toSeq.asJava)
          folderTeam.setRoles(roleSet.toSeq.asJava)
          folderTeam.setPermissions(permissionSet.toSeq.asJava)
      }
    })

    folderTeamsMap.values.toSeq
  }

  private def resetTaskTeam(templateId: String): Unit = {
    val template: Release = releaseRepository.findById(templateId)
    template.getAllTasks.forEach(task => {
      taskService.applyNewTeam(null, task, false)
    })
  }

  private def createDefaultTeamsForTheFolder(folder: Folder, parentId: String): List[XLReleaseEvent] = {
    val currentUser: String = getAuthenticatedUserName

    def newTeam(): Team = Type.valueOf(classOf[Team]).getDescriptor.newInstance(null)

    val folderOwner = newTeam()
    folderOwner.setTeamName(FOLDER_OWNER_TEAMNAME)
    folderOwner.setPermissions(XLReleasePermissions.getFolderPermissions.asScala.filterNot(_ == EDIT_FOLDER_TEAMS.getPermissionName).asJava)
    folderOwner.getPermissions.addAll(XLReleasePermissions.getReleaseGroupPermissions)
    folderOwner.getPermissions.addAll(XLReleasePermissions.getDeliveryPermissions)
    folderOwner.getPermissions.addAll(XLReleasePermissions.getDashboardPermissions)

    val templateOwner = newTeam()
    templateOwner.setTeamName(TEMPLATE_OWNER_TEAMNAME)
    templateOwner.setPermissions(XLReleasePermissions.getTemplateOnlyPermissions)
    templateOwner.getPermissions.add(VIEW_FOLDER.getPermissionName)
    templateOwner.getPermissions.add(VIEW_TRIGGER.getPermissionName)

    val releaseAdmin = newTeam()
    val releaseAdminPermissions = XLReleasePermissions.getReleasePermissions.asScala
      .concat(XLReleasePermissions.getTriggerPermissions.asScala)
      .concat(Seq(VIEW_FOLDER.getPermissionName))

    releaseAdmin.setTeamName(RELEASE_ADMIN_TEAMNAME)
    releaseAdmin.addPermissions(releaseAdminPermissions.toArray)

    if (currentUser != null) {
      folderOwner.addMember(currentUser)
      templateOwner.addMember(currentUser)
      releaseAdmin.addMember(currentUser)
    }

    val result = teamService.saveTeamsToPlatformWithoutPublishing(folder.getId, Seq(folderOwner, templateOwner, releaseAdmin).asJava, true)
    result._2.asScala.toList
  }

  def getNonInheritedFolderReferences(folderId: String, newfolderId: String) = {
    val releases: mutable.Seq[Release] = releaseRepository.search(ReleaseSearchByParams(
      Page(0, Integer.MAX_VALUE, 0),
      statuses = Array.empty,
      folderId = byAncestor(folderId)
    )).asScala

    val res: mutable.Set[BaseConfiguration] = mutable.Set()

    releases.foreach { release =>
      val refs = CiHelper.getExternalReferences(release).asScala
      refs.filter(c => c.isInstanceOf[BaseConfiguration])
        .map(c => configurationRepository.read(c.getId).asInstanceOf[BaseConfiguration])
        .foreach { c =>
          if (!(c.getFolderId == null || // is global OK
            newfolderId.startsWith(c.getFolderId) || // is inherited from parent in scope of destination OK
            c.getFolderId.equals(folderId))) { // will move along because in same folder OK
            res.add(c)
          }
        }
    }
    res
  }

  private def checkFolderCanBeMoved(folderId: String, newfolderIdOption: Option[String], operation: String): Unit = {
    val activeReleases = getRunningReleases(folderId)
    val autoStartPendingReleases = getAutoStartPendingReleases(folderId)

    val invalidRefs: collection.Set[BaseConfiguration] = newfolderIdOption match {
      case Some(newFolderId) => getNonInheritedFolderReferences(folderId, newFolderId)
      case None => Set.empty
    }

    if (activeReleases.nonEmpty || autoStartPendingReleases.nonEmpty || invalidRefs.nonEmpty) {
      logger.warn(s"Tried to $operation folder $folderId, which has active releases ${activeReleases.map(_.getId).mkString(", ")}" +
        s" or pending releases with auto start option enabled: ${autoStartPendingReleases.map(_.getId).mkString(", ")}")

      val details: String = createDetails(activeReleases, autoStartPendingReleases, invalidRefs)

      throw new ItemInUseException(s"You cannot $operation this folder. The folder or its subfolders contain $details")
    }
  }

  private def createDetails(activeReleases: Seq[Release], autoStartPendingReleases: Seq[Release], invalidRefs: collection.Set[BaseConfiguration]) = {
    var details = ""
    if (activeReleases.nonEmpty) {
      details += s"active releases: ${activeReleases.map(_.getTitle).mkString("\"", "\", \"", "\"")}"
    }
    if (autoStartPendingReleases.nonEmpty) {
      details += s"pending releases with auto start: ${autoStartPendingReleases.map(_.getTitle).mkString("\"", "\", \"", "\"")}"
    }
    if (invalidRefs.nonEmpty) {
      details += s"configuration references not inherited by the destination folder: ${invalidRefs.map(_.getTitle).mkString("\"", "\", \"", "\"")}"
    }
    details
  }

  private def checkTemplateCanBeMoved(templateId: String, folderId: String): Unit = {
    val template = releaseRepository.findById(templateId)
    val refs = CiHelper.getExternalReferences(template).asScala

    if (!refs.filter(c => c.isInstanceOf[BaseConfiguration])
      .map(c => configurationRepository.read(c.getId).asInstanceOf[BaseConfiguration])
      .forall(c => c.getFolderId == null || folderId.startsWith(c.getFolderId))) {
      throw new ItemInUseException("You cannot move this template. It contains references to configurations that are " +
        "not global or not inherited by the destination folder.")
    }
  }

  private def getRunningReleases(folderId: String): Seq[Release] = {
    releaseRepository.search(ReleaseSearchByParams(
      page = Page(0, 15, 1),
      statuses = ACTIVE_STATUSES,
      folderId = byAncestor(folderId)
    )).asScala.toSeq
  }

  private def getAutoStartPendingReleases(folderId: String): Seq[Release] = {
    releaseRepository.search(ReleaseSearchByParams(
      page = Page(0, 15, 1),
      statuses = Array(PLANNED),
      folderId = byAncestor(folderId),
      autoStart = true)
    ).asScala.toSeq
  }
}
