package com.xebialabs.xlrelease.scheduler.logs

import akka.actor.ActorRef
import com.xebialabs.xlrelease.domain.events.TaskJobExecutedEvent
import com.xebialabs.xlrelease.events.{EventListener, Subscribe}
import com.xebialabs.xlrelease.repository.Ids
import com.xebialabs.xlrelease.runner.domain.JobId
import com.xebialabs.xlrelease.scheduler.events.JobFinishedEvent
import com.xebialabs.xlrelease.scheduler.logs.ExecutionLogWatchActor._
import com.xebialabs.xlrelease.scheduler.logs.TaskExecutionRepository.{ByTaskId, FinishExecution, UpdateWithLogEntry}
import com.xebialabs.xlrelease.scheduler.storage.spring.StorageConfiguration.URI_SCHEME_LOCAL_STORAGE
import com.xebialabs.xlrelease.service.BroadcastService
import com.xebialabs.xlrelease.storage.domain.LogEntry
import com.xebialabs.xlrelease.storage.service.StorageService
import com.xebialabs.xlrelease.support.akka.spring.SpringExtension
import com.xebialabs.xlrelease.views.TaskExecutionLogView
import grizzled.slf4j.Logging
import org.apache.commons.codec.digest.DigestUtils
import org.springframework.stereotype.Service

import java.io.OutputStream
import java.time.Instant
import javax.ws.rs.sse.{Sse, SseEventSink}
import scala.jdk.CollectionConverters._

@Service
@EventListener
class TaskExecutionLogService(springExtension: SpringExtension,
                              broadcastService: BroadcastService,
                              storageService: StorageService,
                              taskExecutionRepository: TaskExecutionRepository)
  extends Logging {

  private lazy val executionLogWatcherActorRef: ActorRef = springExtension.actorOf(classOf[ExecutionLogWatcherSupervisor], s"execution-log-watchers")

  def watch(taskId: String, executionId: String, sink: SseEventSink, sse: Sse): Unit = {
    executionLogWatcherActorRef ! StartWatch(taskId, executionId, sink, sse)
  }

  def fetch(taskId: String, executionId: String, outputStream: OutputStream, lastJob: JobId, lastChunk: Long): Unit = {
    TaskExecutionLog(taskId, executionId).fetch(storageService, outputStream, lastJob, lastChunk)
  }

  def fetchAllExecutions(taskId: String): java.util.List[TaskExecutionLogView] = {
    taskExecutionRepository.find(ByTaskId(taskId)).zipWithIndex.map { case (row, index) =>
      val view = new TaskExecutionLogView()
      view.setId(row.executionId)
      view.setExecutionNo(index + 1)
      view.setLastJob(row.lastJob)
      view.setLastChunk(row.lastChunk)
      val modifiedDate = Option(row.lastModifiedDate) match {
        case Some(date) => date.toEpochMilli
        case None => Instant.now().toEpochMilli
      }
      view.setModifiedDate(modifiedDate)
      Option(row.endDate).foreach(date => view.setEndDate(date.toEpochMilli))
      view
    }.asJava
  }

  def log(logEntry: LogEntry): Unit = {
    val workerLogEntry = logEntry.copy(uriScheme = URI_SCHEME_LOCAL_STORAGE)
    val storedEntryUri = storageService.store(workerLogEntry)
    taskExecutionRepository.update(UpdateWithLogEntry(workerLogEntry))
    broadcastService.broadcast(TaskLogCreated(logEntry.taskId, logEntry.executionId, storedEntryUri), true)
  }

  def getTaskExecutionEntry(taskId: String, executionId: String): Option[TaskExecutionEntry] = {
    taskExecutionRepository.read(taskId, executionId)
  }

  @Subscribe
  def onJobFinished(event: JobFinishedEvent): Unit = {
    val jobId = event.jobId
    val executionId = event.executionId
    logger.debug(s"finishing job $jobId")
    executionLogWatcherActorRef ! Check(executionId)
  }

  @Subscribe
  def onTaskLogCreated(taskLogEvent: TaskLogCreated): Unit = {
    logger.debug(s"processing log event $taskLogEvent")
    executionLogWatcherActorRef ! NewEntry(taskLogEvent.executionId, taskLogEvent.uri)
  }

  @Subscribe
  def onTaskExecutionDone(event: TaskJobExecutedEvent): Unit = {
    val finishExecution = FinishExecution(event.taskId, event.executionId, endDate = Instant.now())
    taskExecutionRepository.update(finishExecution)
  }
}

object TaskExecutionLogService {
  // TODO hash comes from TaskPersistence, move it to some more common place (domain, utils...)
  def hash(taskId: String): String = DigestUtils.sha256Hex(Ids.getFolderlessId(taskId))
}
