package com.xebialabs.xlrelease.spring.configuration;

import java.util.Arrays;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Profile;
import org.springframework.security.access.vote.AuthenticatedVoter;
import org.springframework.security.access.vote.UnanimousBased;
import org.springframework.security.authentication.AuthenticationManager;
import org.springframework.security.core.session.SessionRegistry;
import org.springframework.security.core.userdetails.UserDetailsService;
import org.springframework.security.web.authentication.logout.LogoutFilter;
import org.springframework.security.web.authentication.logout.LogoutSuccessEventPublishingLogoutHandler;
import org.springframework.security.web.authentication.logout.SecurityContextLogoutHandler;
import org.springframework.security.web.authentication.rememberme.RememberMeAuthenticationFilter;
import org.springframework.security.web.authentication.session.CompositeSessionAuthenticationStrategy;
import org.springframework.security.web.authentication.session.ConcurrentSessionControlAuthenticationStrategy;
import org.springframework.security.web.authentication.session.RegisterSessionAuthenticationStrategy;
import org.springframework.security.web.authentication.session.SessionFixationProtectionStrategy;
import org.springframework.security.web.authentication.www.BasicAuthenticationFilter;
import org.springframework.security.web.csrf.CsrfLogoutHandler;
import org.springframework.security.web.firewall.HttpStatusRequestRejectedHandler;
import org.springframework.security.web.firewall.RequestRejectedHandler;
import org.springframework.security.web.session.ConcurrentSessionFilter;

import com.xebialabs.deployit.ServerConfiguration;
import com.xebialabs.deployit.plumbing.LogbackAccessSecurityAttributesSaveFilter;
import com.xebialabs.deployit.plumbing.authentication.*;
import com.xebialabs.xlrelease.authentication.XlPersistentTokenRememberMeServices;
import com.xebialabs.xlrelease.config.XlrConfig;
import com.xebialabs.xlrelease.repository.CustomPersistentTokenRepository;
import com.xebialabs.xlrelease.security.filter.PATAuthenticationFilter;
import com.xebialabs.xlrelease.security.filter.RunnerAuthenticationFilter;
import com.xebialabs.xlrelease.security.web.csrf.ReleaseCookieCsrfTokenRepository;
import com.xebialabs.xlrelease.service.UserLastActiveActorService;
import com.xebialabs.xlrelease.service.UserProfileService;

@Configuration
public class ReleaseSecurityConfiguration {

    private final AuthenticationManager authenticationManager;

    private final UserDetailsService userDetailsService;

    private final UserProfileService userProfileService;

    private final UserLastActiveActorService userLastActiveActorService;

    private final CustomPersistentTokenRepository customPersistentTokenRepository;

    private final ServerConfiguration serverConfiguration;

    private final SessionRegistry sessionRegistry;

    private final XlrConfig xlrConfig;

    @Autowired
    public ReleaseSecurityConfiguration(AuthenticationManager authenticationManager,
                                        UserDetailsService userDetailsService,
                                        UserProfileService userProfileService,
                                        UserLastActiveActorService userLastActiveActorService,
                                        CustomPersistentTokenRepository customPersistentTokenRepository,
                                        ServerConfiguration serverConfiguration,
                                        SessionRegistry sessionRegistry,
                                        XlrConfig xlrConfig) {
        this.authenticationManager = authenticationManager;
        this.userDetailsService = userDetailsService;
        this.userProfileService = userProfileService;
        this.userLastActiveActorService = userLastActiveActorService;
        this.customPersistentTokenRepository = customPersistentTokenRepository;
        this.serverConfiguration = serverConfiguration;
        this.sessionRegistry = sessionRegistry;
        this.xlrConfig = xlrConfig;
    }

    @Bean
    public AuthenticatedVoter authenticatedVoter() {
        return new AuthenticatedVoter();
    }

    @Bean
    public LoginPermissionVoter loginPermissionVoter() {
        return new LoginPermissionVoter(userProfileService, userLastActiveActorService);
    }

    @Bean
    public UnanimousBased unanimousBased() {
        return new UnanimousBased(Arrays.asList(authenticatedVoter(), loginPermissionVoter()));
    }

    @Bean
    public Http401AuthenticationEntryPoint http401AuthenticationEntryPoint() {
        Http401AuthenticationEntryPoint authenticationEntryPoint = new Http401AuthenticationEntryPoint();
        authenticationEntryPoint.setRealmName("XLRelease");
        return authenticationEntryPoint;
    }

    @Bean
    public BasicAuthenticationFilter basicAuthenticationFilter() {
        return new BasicAuthenticationFilter(authenticationManager, http401AuthenticationEntryPoint());
    }

    @Bean
    public PATAuthenticationFilter patAuthenticationFilter() {
        return new PATAuthenticationFilter(authenticationManager, http401AuthenticationEntryPoint(), "x-release-personal-token");
    }

    @Bean
    @Profile(XlrProfiles.DEFAULT_AUTH)
    public UsernamePasswordAuthenticationFilter formLoginFilter() {
        UsernamePasswordAuthenticationFilter authenticationFilter = new UsernamePasswordAuthenticationFilter();
        authenticationFilter.setAuthenticationManager(authenticationManager);
        authenticationFilter.setSessionAuthenticationStrategy(sessionAuthenticationStrategy());
        authenticationFilter.setRememberMeServices(rememberMeServices());
        authenticationFilter.setAuthenticationSuccessHandler(withoutRedirectLoginSuccessHandler());
        authenticationFilter.setAuthenticationFailureHandler(releaseAuthenticationFailureHandler());
        return authenticationFilter;
    }

    @Bean
    public ConcurrentSessionFilter concurrentSessionFilter() {
        ConcurrentSessionFilter sessionFilter = new ConcurrentSessionFilter(sessionRegistry);
        sessionFilter.setLogoutHandlers(Arrays.asList(securityContextLogoutHandler(), sessionExpiredLogoutHandler()));
        return sessionFilter;
    }

    @Bean
    public LogbackAccessSecurityAttributesSaveFilter logbackAccessSecurityAttributesSaveFilter() {
        return new LogbackAccessSecurityAttributesSaveFilter();
    }

    @Bean
    @Profile(XlrProfiles.DEFAULT_AUTH)
    public RememberMeAuthenticationFilter rememberMeFilter() {
        return new RememberMeAuthenticationFilter(authenticationManager, rememberMeServices());
    }

    @Bean
    @Profile(XlrProfiles.DEFAULT_AUTH)
    public LogoutFilter logoutFilter() {
        return new LogoutFilter(withoutRedirectLogoutSuccessHandler(), rememberMeServices(), csrfLogoutHandler(), logoutEventPublishingLogoutHandler());
    }

    @Bean
    @Profile(XlrProfiles.DEFAULT_AUTH)
    public XlPersistentTokenRememberMeServices rememberMeServices() {
        XlPersistentTokenRememberMeServices tokenRememberMeServices = new XlPersistentTokenRememberMeServices(
                xlrConfig.server_session_remember_me_key(), userDetailsService, customPersistentTokenRepository, serverConfiguration);
        tokenRememberMeServices.setParameter("rememberMe");
        tokenRememberMeServices.setCookieName("REMEMBER_ME_XLR");
        tokenRememberMeServices.setTokenValiditySeconds(xlrConfig.server_session_remember_me_token_validity_seconds());
        return tokenRememberMeServices;
    }

    // session management
    @Bean
    public CompositeSessionAuthenticationStrategy sessionAuthenticationStrategy() {
        return new CompositeSessionAuthenticationStrategy(
                Arrays.asList(concurrentSessionControlAuthenticationStrategy(),
                        sessionFixationProtectionStrategy(),
                        registerSessionAuthenticationStrategy()));
    }

    @Bean
    public ConcurrentSessionControlAuthenticationStrategy concurrentSessionControlAuthenticationStrategy() {
        ConcurrentSessionControlAuthenticationStrategy authenticationStrategy = new ConcurrentSessionControlAuthenticationStrategy(sessionRegistry);
        authenticationStrategy.setMaximumSessions(xlrConfig.server_session_maximum_sessions());
        authenticationStrategy.setExceptionIfMaximumExceeded(xlrConfig.server_session_exception_if_maximum_exceeded());
        return authenticationStrategy;
    }

    @Bean
    public SessionFixationProtectionStrategy sessionFixationProtectionStrategy() {
        return new SessionFixationProtectionStrategy();
    }

    @Bean
    public RegisterSessionAuthenticationStrategy registerSessionAuthenticationStrategy() {
        return new RegisterSessionAuthenticationStrategy(sessionRegistry);
    }

    // Handlers
    @Bean
    public SecurityContextLogoutHandler securityContextLogoutHandler() {
        return new SecurityContextLogoutHandler();
    }

    @Bean
    public SessionExpiredLogoutHandler sessionExpiredLogoutHandler() {
        return new SessionExpiredLogoutHandler();
    }

    @Bean
    @Profile(XlrProfiles.DEFAULT_AUTH)
    public WithoutRedirectLogoutSuccessHandler withoutRedirectLogoutSuccessHandler() {
        return new WithoutRedirectLogoutSuccessHandler();
    }

    @Bean
    @Profile(XlrProfiles.DEFAULT_AUTH)
    public WithoutRedirectLoginSuccessHandler withoutRedirectLoginSuccessHandler() {
        return new WithoutRedirectLoginSuccessHandler();
    }

    @Bean
    @Profile(XlrProfiles.DEFAULT_AUTH)
    public ReleaseAuthenticationFailureHandler releaseAuthenticationFailureHandler() {
        return new ReleaseAuthenticationFailureHandler();
    }

    @Bean
    @Profile(XlrProfiles.DEFAULT_AUTH)
    public LogoutSuccessEventPublishingLogoutHandler logoutEventPublishingLogoutHandler() {
        return new LogoutSuccessEventPublishingLogoutHandler();
    }

    // CSRF
    @Bean
    public ReleaseCookieCsrfTokenRepository tokenRepository() {
        ReleaseCookieCsrfTokenRepository csrfTokenRepository = new ReleaseCookieCsrfTokenRepository();
        csrfTokenRepository.setCookieHttpOnly(false);
        csrfTokenRepository.setSameSite(xlrConfig.server_http_cookie_sameSite().getAttributeValue());
        if (!serverConfiguration.isSsl()) {
            csrfTokenRepository.setSecure(serverConfiguration.isSecureCookieEnabled());
        }
        return csrfTokenRepository;
    }

    @Bean
    public CsrfSecurityRequestMatcher csrfSecurityRequestMatcher() {
        return new CsrfSecurityRequestMatcher();
    }

    @Bean
    @Profile(XlrProfiles.DEFAULT_AUTH)
    public CsrfLogoutHandler csrfLogoutHandler() {
        return new CsrfLogoutHandler(tokenRepository());
    }

    @Bean
    public RequestRejectedHandler requestRejectedHandler() {
        return new HttpStatusRequestRejectedHandler();
    }

    @Bean
    public RunnerAuthenticationFilter runnerAuthenticationFilter() {
        return new RunnerAuthenticationFilter(this.authenticationManager, http401AuthenticationEntryPoint(), "x-runner-auth-token");
    }

}
