package com.xebialabs.xlrelease.service

import com.xebialabs.deployit.checks.Checks.{checkArgument, checkNotNull}
import com.xebialabs.deployit.plugin.api.reflect.{PropertyDescriptor, PropertyKind, Type}
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem
import com.xebialabs.deployit.util.PasswordEncrypter
import com.xebialabs.xlrelease.api.v1.forms.VariableOrValue
import com.xebialabs.xlrelease.domain.variables.reference.{UsagePoint, VariableMappingUsagePoint, VariableReference}
import com.xebialabs.xlrelease.domain.variables.{GlobalVariables, PasswordStringVariable, Variable}
import com.xebialabs.xlrelease.domain.{Configuration, ExternalVariableServer}
import com.xebialabs.xlrelease.repository.CiProperty
import com.xebialabs.xlrelease.service.ConfigurationVariableService.{getUsagePointsByVars, replaceFromUsagePoints}
import com.xebialabs.xlrelease.variable.{VariableHelper, VariablesGraphResolver}
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Component

import scala.collection.mutable
import scala.compat.java8.OptionConverters._
import scala.jdk.CollectionConverters._

@Component
class ConfigurationVariableService @Autowired()(variableService: VariableService, folderVariableService: FolderVariableService) {

  def resolve(configs: Iterable[Configuration], globalVariables: GlobalVariables = null): Unit = {
    val configsWithVarMapping = configs.filter(_.hasVariableMapping)
    if (configsWithVarMapping.nonEmpty) {
      configsWithVarMapping.foreach { conf =>
        checkArgument(!conf.isInstanceOf[ExternalVariableServer], "External variable servers can't use variables")
      }
      val configsByFolder = configsWithVarMapping.groupBy(_.getFolderId)
      val globalVars = {
        if (globalVariables == null) variableService.findGlobalVariablesOrEmpty() else globalVariables
      }.getVariables.asScala.toSeq
      // todo if e.g. custom script task overrides fields they do not need to be resolved
      configsByFolder.foreachEntry { (folderId, confs) =>
        val usagePointsByVars = getUsagePointsByVars(confs)

        if (usagePointsByVars.nonEmpty) {
          val folderVars = if (folderId != null) folderVariableService.getAllFromAncestry(folderId).getVariables.asScala else Seq.empty

          replaceFromUsagePoints(usagePointsByVars, globalVars ++ folderVars)
        }
      }
    }
  }

  def resolve(config: Configuration): Unit = {
    resolve(Seq(config))
  }

  def resolveFromCi[T <: ConfigurationItem](ci: T, globalVariables: GlobalVariables = null)
                                           (getDescriptors: T => Iterable[PropertyDescriptor] = (_ci: T) =>
                                             _ci.getType.getDescriptor.getPropertyDescriptors.asScala
                                           ): Unit = {
    val confType = Type.valueOf(classOf[Configuration])
    val configs = getDescriptors(ci).view
      .filter(_.getKind == PropertyKind.CI)
      .filter(_.getReferencedType.isSubTypeOf(confType))
      .map(_.get(ci))
      .filter(_ != null)
      .map(_.asInstanceOf[Configuration])

    resolve(configs, globalVariables)
  }
}

object ConfigurationVariableService {
  def getUsagePointsByVars(configs: Iterable[Configuration]): Map[String, Set[UsagePoint]] = {
    configs
      .flatMap(collectVariablesInVariableMappings)
      .groupMapReduce { case (varKey, _) =>
        VariableHelper.withoutVariableSyntax(varKey)
      }(_._2.getUsagePoints.asScala.toSet)(_ ++ _)
  }

  // Gets all used variables by configuration
  // {varKey -> VariableReference(varKey, usageType, [UsagePoint(ci, propertyName, CiProperty)]}
  def collectVariablesInVariableMappings(conf: Configuration): mutable.Map[String, VariableReference] = {
    val variableReferences = mutable.TreeMap.empty[String, VariableReference]
    conf.getVariableMapping.asScala.foreach { case (fqPropertyName, varKey) if varKey != null =>
      CiProperty.of(conf, fqPropertyName).asScala.map { ciProperty =>
        new VariableMappingUsagePoint(conf, fqPropertyName, ciProperty)
      }.foreach { up =>
        val variablesAtUsagePoint = up.collectVariables()
        variablesAtUsagePoint.forEach { case (varKey, usageType) =>
          val ref = variableReferences.getOrElseUpdate(varKey, new VariableReference(varKey, usageType))
          ref.addUsagePoint(up, usageType)
        }
      }
    }

    variableReferences
  }

  def replaceFromUsagePoint(varKey: String, usagePoint: UsagePoint, availableVars: Seq[Variable]): Unit = {
    replaceFromUsagePoints(Map(varKey -> Set(usagePoint)), availableVars)
  }

  def replaceFromUsagePoints(usagePointsByVars: Map[String, Set[UsagePoint]], availableVars: Seq[Variable]): Unit = {
    val (externalVars, normalVars) = availableVars.partition { variable =>
      variable.isPassword && variable.asInstanceOf[PasswordStringVariable].getExternalVariableValue != null
    }

    val externalVariablesByKey = VariableHelper.indexByKey(externalVars.asJava)
    val resolvedVars = VariablesGraphResolver.resolve(normalVars.toSet)

    // group variables by their server so they can be bulk resolved
    val serverMap = mutable.Map.empty[ExternalVariableServer, Seq[PasswordStringVariable]]
    // freeze variable mapping for normal variables and collect external variables
    usagePointsByVars.foreachEntry { (varKey, usagePoints) =>
      resolvedVars.get(varKey) match {
        case Some(graphVar) =>
          checkArgument(graphVar.resolved, s"Unable to resolve variable '$varKey'=${graphVar.value} with Id '${graphVar.variable.getId}'")
          val value = if (graphVar.variable.isPassword) {
            PasswordEncrypter.getInstance.ensureEncrypted(graphVar.variable.getValueAsString)
          } else {
            graphVar.value
          }
          usagePoints.foreach(_.replaceVariable(graphVar.variable, new VariableOrValue(null, value)))
        case None =>
          val externalVar = externalVariablesByKey.get(varKey)
          checkNotNull(externalVar, s"Unable to find variable '$varKey'")
          val passwordVar = externalVar.asInstanceOf[PasswordStringVariable]
          val server = passwordVar.getExternalVariableValue.getServer
          checkNotNull(server, s"Unable to load configuration for variable '${passwordVar.getKey}' with Id '${passwordVar.getId}'")
          // don't let external variable servers reference variables for now, doesn't work in as-code anyway
          // later: construct dependency graph with variables and servers, make sure cycle is detected between servers and variables
          checkArgument(server.getVariableMapping.isEmpty, "External variable servers can't use variables")
          serverMap.put(server, serverMap.getOrElse(server, Seq.empty).prepended(passwordVar))
      }
    }
    // resolve external variables and freeze variable mapping
    serverMap.foreachEntry { (server, vars) =>
      val externalValues = server.lookup(vars.asJava)
      externalValues.forEach { (varKey, varValue) =>
        usagePointsByVars(varKey).foreach(_.replaceVariable(
          externalVariablesByKey.get(varKey),
          new VariableOrValue(null, PasswordEncrypter.getInstance.ensureEncrypted(varValue))
        ))
      }
    }
  }
}