package com.xebialabs.xlrelease.scheduler.logs

import com.xebialabs.xlrelease.repository.Ids.getFolderlessId
import com.xebialabs.xlrelease.scheduler.storage.spring.StorageConfiguration.URI_SCHEME_LOCAL_STORAGE
import com.xebialabs.xlrelease.storage.domain.{JobEntryRef, LogDirectoryRef}
import com.xebialabs.xlrelease.storage.service.StorageService
import com.xebialabs.xlrelease.views.TaskJobLogView
import org.apache.commons.codec.digest.DigestUtils
import org.apache.commons.io.IOUtils

import java.io.{File, InputStream, OutputStream}
import java.net.URI
import scala.util.{Try, Using}

case class TaskJobLog(taskId: String, jobId: Option[Long]) {

  def fetch(storageService: StorageService): LazyList[Try[InputStream]] = {
    if (jobId.isEmpty) {
      throw new IllegalStateException("jobId is empty, can't fetch logs without it")
    }
    for {
      is <- LazyList.from(1).map { chunk =>
        val jobUri = taskJobChunkUri(chunk)
        val jobEntryRef = JobEntryRef(jobId.get, jobUri)
        Try(storageService.get(jobEntryRef))
      }.takeWhile(_.isSuccess)
    } yield is
  }

  def fetch(storageService: StorageService, outputStream: OutputStream): Unit = {
    for {
      is <- fetch(storageService)
      iss <- is
    } Try {
      Using.resource(iss) { content => IOUtils.copy(content, outputStream) }
    }
  }

  def fetchJobsIds(storageService: StorageService): Seq[TaskJobLogView] = {
    storageService.list(LogDirectoryRef(taskId, taskJobUri())).map(u => {
      val directory = new File(u)
      val taskJobLogView = new TaskJobLogView()
      taskJobLogView.setId(directory.getName)
      taskJobLogView.setModifiedDate(directory.lastModified())
      taskJobLogView
    })
  }

  private def taskJobChunkUri(chunk: Int): URI = {
    val taskIdHash = getTaskIdHash
    URI.create(s"$URI_SCHEME_LOCAL_STORAGE:///jobs/$taskIdHash/${jobId.get}/$chunk")
  }

  private def taskJobUri(): URI = {
    val taskIdHash = getTaskIdHash
    URI.create(s"$URI_SCHEME_LOCAL_STORAGE:///jobs/$taskIdHash")
  }

  private def getTaskIdHash: String = {
    import TaskJobLog.TaskIdHash
    taskId.hash()
  }

}

object TaskJobLog {
  implicit class TaskIdHash(taskId: String) {
    def hash(): String = DigestUtils.sha256Hex(getFolderlessId(taskId))
  }
}