package com.xebialabs.xlrelease.scheduler.logs

import akka.actor.{Actor, ActorLogging}
import com.xebialabs.xlrelease.runner.domain.JobId
import com.xebialabs.xlrelease.scheduler.logs.JobLogWatchActor.{NewEntry, WatcherMsg}
import com.xebialabs.xlrelease.scheduler.logs.JobLogWatcherSupervisor.JobLogWatchTerminated
import com.xebialabs.xlrelease.scheduler.storage.spring.StorageConfiguration.URI_SCHEME_LOCAL_STORAGE
import com.xebialabs.xlrelease.scheduler.storage.{ReleaseStorageEntryCreated, ReleaseStorageEvent}
import com.xebialabs.xlrelease.support.akka.spring.{SpringActor, SpringExtension}
import org.springframework.util.AntPathMatcher

import scala.collection.mutable

@SpringActor
class JobLogWatcherSupervisor extends Actor with ActorLogging {
  private val watchedJobIds: mutable.Set[JobId] = mutable.Set()

  // list of watched job ids...
  override def receive: Receive = {
    case storageEvent: ReleaseStorageEvent => handleStorageEvent(storageEvent)
    case JobLogWatchTerminated(jobId) => watchedJobIds -= jobId
    case m: WatcherMsg =>
      val actorName = s"job-log-watcher-${m.jobId}"
      val actorRef = context.child(actorName).getOrElse {
        SpringExtension.childActorOf(classOf[JobLogWatchActor], actorName)(context)
      }
      context.watchWith(actorRef, JobLogWatchTerminated(m.jobId))
      watchedJobIds.add(m.jobId)
      actorRef.forward(m)
  }

  private def handleStorageEvent(event: ReleaseStorageEvent): Unit = event match {
    case ReleaseStorageEntryCreated(newUri) =>
      watchedJobIds.foreach { jobId =>
        // TODO hardcoded scheme - storage should be "configurable"
        val pathMatcher = new AntPathMatcher()
        pathMatcher.setCachePatterns(false)
        if (newUri.getScheme == URI_SCHEME_LOCAL_STORAGE && pathMatcher.`match`(s"/jobs/*/$jobId/**", newUri.getPath)) {
          self ! NewEntry(jobId, newUri)
        }
      }
  }
}

object JobLogWatcherSupervisor {
  case class JobLogWatchTerminated(jobId: JobId)
}