package com.xebialabs.xlrelease.scheduler.logs

import akka.actor.{Actor, ActorLogging, ReceiveTimeout}
import com.xebialabs.xlrelease.domain.status.TaskStatus._
import com.xebialabs.xlrelease.repository.TaskRepository
import com.xebialabs.xlrelease.runner.domain.JobId
import com.xebialabs.xlrelease.scheduler.logs.JobLogWatchActor._
import com.xebialabs.xlrelease.storage.domain.JobEntryRef
import com.xebialabs.xlrelease.storage.service.StorageService
import com.xebialabs.xlrelease.support.akka.spring.SpringActor
import org.apache.commons.io.IOUtils

import java.net.URI
import java.time.{Duration, Instant}
import javax.ws.rs.sse.{OutboundSseEvent, Sse, SseEventSink}
import scala.collection.mutable
import scala.collection.mutable.ArrayBuffer
import scala.concurrent.duration.DurationInt
import scala.util.{Try, Using}

@SpringActor
class JobLogWatchActor(storageService: StorageService, taskRepository: TaskRepository) extends Actor with ActorLogging {
  var jobId: JobId = _
  var taskId: String = _
  // if this actor is stopped just close sse and do not restart it
  var sinks: mutable.Buffer[SseEventSink] = ArrayBuffer[SseEventSink]()
  var sse: Sse = _
  var lastEventTimestamp: Instant = Instant.now()

  context.setReceiveTimeout(5.seconds)

  override def receive: Receive = {
    case m: WatcherMsg => handleWatchMsgs(m)
    case ReceiveTimeout => self ! Check(jobId)
  }

  override def postStop(): Unit = {
    this.sinks.foreach(s => Try(s.close()))
  }

  //noinspection ScalaStyle
  private def handleWatchMsgs(watcherMsg: WatcherMsg): Unit = watcherMsg match {
    case StartWatch(taskId, jobId, fromStart, sseEventSink, sse) =>
      this.jobId = jobId
      this.taskId = taskId
      this.sinks += sseEventSink
      this.sse = sse
      streamChunks(fromStart, sseEventSink)
    case StopWatching(_) =>
      this.sinks.foreach { s => if (!s.isClosed) s.close() }
      self ! Check(jobId)
    case Check(id) =>
      val (closedSinks, openSinks) = sinks.partition(s => s.isClosed)
      sinks.subtractAll(closedSinks)
      // should unsubscribe... any msg that arrives will be sent to dead letter
      if (openSinks.isEmpty) {
        stop()
      } else {
        // if there was no new entry for certain amount of time ... send ping so we would close closed sinks
        // check stored instant of the last processed entry, and if it's less than
        val durationSinceLastEvent = Duration.between(lastEventTimestamp, Instant.now()).toSeconds
        val taskStatus = taskRepository.getStatus(taskId)
        if (taskStatus.isOneOf(COMPLETED, SKIPPED, FAILED)) {
          self ! StopWatching(id)
        } else {
          if (durationSinceLastEvent > 10 && durationSinceLastEvent < 60) {
            val ping = this.sse.newEvent(PING_SSE_EVENT, new String(":\n\n"))
            sendEventToAllSinks(ping)
          } else if (durationSinceLastEvent >= 60) {
            self ! StopWatching(id)
          }
        }
      }
    case NewEntry(jobId, newEntryUri) =>
      this.lastEventTimestamp = Instant.now()
      // store instant of the entry into internal state
      val payload = Using.resource(storageService.get(JobEntryRef(jobId, newEntryUri))) { content =>
        IOUtils.toByteArray(content)
      }
      val event = this.sse.newEvent(LOG_CHUNK_ENTRY_CREATED_EVENT, new String(payload))
      sendEventToAllSinks(event)
  }

  private def sendEventToAllSinks(event: OutboundSseEvent): Unit = {
    this.sinks.foreach(sendEventToSink(event))
  }

  private def sendEventToSink(event: OutboundSseEvent)(sink: SseEventSink) = {
    if (!sink.isClosed) {
      import scala.jdk.FutureConverters._
      sink.send(event).asScala.recover { t => {
        log.error(t, "Event send failed")
        sink.close()
      }
      }(scala.concurrent.ExecutionContext.parasitic)
    }
  }

  private def streamChunks(fromStart: Boolean, sink: SseEventSink): Unit = {
    if (fromStart) {
      for {
        is <- TaskJobLog(taskId, Some(jobId)).fetch(storageService)
        iss <- is
      } Try {
        val payload = Using.resource(iss) { content =>
          IOUtils.toByteArray(content)
        }
        val event = this.sse.newEvent(LOG_CHUNK_ENTRY_CREATED_EVENT, new String(payload))
        sendEventToSink(event)(sink)
      }
    }
    self ! Check(jobId)
  }

  private def stop(): Unit = {
    context.stop(self)
  }
}

object JobLogWatchActor {

  final val LOG_CHUNK_ENTRY_CREATED_EVENT: String = "log-chunk-entry-created"

  final val PING_SSE_EVENT: String = "ping"

  sealed trait WatcherMsg {
    def jobId: JobId
  }

  case class StartWatch(taskId: String, jobId: JobId, fromStart: Boolean, sink: SseEventSink, sse: Sse) extends WatcherMsg

  case class StopWatching(jobId: JobId) extends WatcherMsg

  case class Check(jobId: JobId) extends WatcherMsg

  case class NewEntry(jobId: JobId, newEntryUri: URI) extends WatcherMsg
}
