package com.xebialabs.xlrelease.scheduler

import com.codahale.metrics.annotation.Timed
import com.xebialabs.xlrelease.actors.ActorSystemHolder
import com.xebialabs.xlrelease.runner.domain.{JobId, RunnerId}
import com.xebialabs.xlrelease.scheduler.converters.TaskJobConverter
import com.xebialabs.xlrelease.scheduler.domain.LocalJobRunner
import com.xebialabs.xlrelease.scheduler.events.{JobCreatedEvent, JobFinishedEvent, JobReservedEvent}
import com.xebialabs.xlrelease.scheduler.repository.{ConfirmJobExecution, DeleteById, JobRepository, ReserveJob}
import com.xebialabs.xlrelease.service.BroadcastService
import grizzled.slf4j.Logging
import org.springframework.context.annotation.{Lazy, Scope, ScopedProxyMode}
import org.springframework.stereotype.Component

import java.util.concurrent.DelayQueue
import java.util.concurrent.atomic.AtomicBoolean
import java.util.function.Predicate
import scala.util.{Failure, Success, Try}


@Component
@Scope(proxyMode = ScopedProxyMode.TARGET_CLASS)
class DefaultJobQueue(val taskJobConverter: TaskJobConverter,
                      val jobRepository: JobRepository,
                      broadcastService: BroadcastService,
                      @Lazy
                      val actorSystemHolder: ActorSystemHolder)
  extends Logging with NodeId {

  private val _localJobs: DelayQueue[Job] = new DelayQueue[Job]()
  private val _isRunning = new AtomicBoolean(true)

  @Timed
  def submit(job: Job): Unit = {
    logger.debug(s"submitted job $job")
    job match {
      case job: TaskJob[_] =>
        job.setId(jobRepository.create(JobRow(job).copy(node = nodeId)).id)
      case _ => () // special case when we send a msg that does not have task id (i.e. StopWorkerThread)
    }
    addToLocalJobs(job, broadcastJob = job.broadcast)
  }

  def submitExisting(job: Job): Unit = {
    logger.debug(s"submitted existing job $job")
    cancelIf(_.id == job.id)
    addToLocalJobs(job, broadcastJob = job.broadcast)
  }

  def submitBroadcasted(job: Job): Unit = {
    logger.debug(s"submitted broadcasted job $job")
    cancelIf(_.id == job.id)
    addToLocalJobs(job, broadcastJob = false)
  }

  @Timed
  def replace(job: Job): Unit = {
    logger.debug(s"replaced job $job")
    job.id = jobRepository.replace(JobRow(job).copy(node = nodeId, runnerId = null)).id
    localJobs.add(job)
  }

  def isRunning(): Boolean = {
    _isRunning.get()
  }

  def localJobs: DelayQueue[Job] = {
    _localJobs
  }

  def reserve(job: Job, runnerId: String = LocalJobRunner.getId()): Job = {
    val lockAttempt = jobRepository.update(ReserveJob(JobRow(job), nodeId, runnerId))
    lockAttempt match {
      case Failure(exception) => throw exception
      case Success(row) =>
        val lockedJob = job match {
          case taskJob: TaskJob[_] => taskJobConverter.fromJobRow(row, Some(taskJob.taskRef)) // row -> taskJob
          case _ => taskJobConverter.fromJobRow(row) // row -> taskJob
        }
        broadcastService.broadcast(JobReservedEvent(lockedJob.id), publishEventOnSelf = false)
        localJobs.remove(job) // safe to remove job that is not there
        lockedJob
    }
  }

  def confirm(runnerId: RunnerId, jobId: JobId): Boolean = {
    val confirmationResult = for {
      jobRow <- Try(jobRepository.read(jobId))
      confirmedRow <- jobRepository.update(ConfirmJobExecution(jobRow, runnerId))
    } yield confirmedRow
    confirmationResult match {
      case Failure(ex) =>
        logger.warn(s"Unable to confirm job $jobId", ex)
        false
      case Success(_) =>
        true
    }
  }

  @Timed
  def finish(jobId: JobId): Unit = {
    logger.debug(s"finishing job $jobId")
    jobRepository.delete(DeleteById(jobId))
    broadcastService.broadcast(JobFinishedEvent(jobId), publishEventOnSelf = true)
  }

  def start(): Unit = {
    logger.debug(s"starting job queue")
    _isRunning.set(true)
  }

  def stop(): Unit = {
    logger.debug(s"stopping job queue")
    _isRunning.set(false)
  }

  def cancelIf(predicate: Predicate[Job]): Unit = {
    localJobs.removeIf(predicate)
  }

  def size(): Int = {
    localJobs.size()
  }

  private def addToLocalJobs(job: Job, broadcastJob: Boolean): Unit = {
    localJobs.add(job)
    if (broadcastJob) {
      broadcastService.broadcast(JobCreatedEvent(job), publishEventOnSelf = false)
    }
  }
}
