package com.xebialabs.xlrelease.repository

import com.xebialabs.deployit.exception.NotFoundException
import com.xebialabs.deployit.repository.ItemConflictException
import com.xebialabs.xlrelease.api.internal.EffectiveSecurityDecorator.EFFECTIVE_SECURITY
import com.xebialabs.xlrelease.api.internal.{DecoratorsCache, InternalMetadataDecoratorService}
import com.xebialabs.xlrelease.domain.folder.Folder
import com.xebialabs.xlrelease.domain.status.ReleaseStatus.ACTIVE_STATUSES
import grizzled.slf4j.Logging

import scala.jdk.CollectionConverters._

trait FolderRepository extends Logging {

  def decoratorService: InternalMetadataDecoratorService

  def create(parentId: String, folder: Folder): Folder

  def delete(folderId: String, deleteRelease: (Int, String) => Unit = (_, _) => ()): Unit

  def rename(folderId: String, newName: String): Folder

  def move(folderId: String, newParentId: String): Folder

  // queries
  def exists(folderId: String): Boolean

  def findSubFolderByTitle(parentId: String, name: String): Option[Folder]

  def isFolderInherited(folderId: String): Boolean

  def getTitle(folderId: String): Option[String]

  def getPath(folderId: String): Seq[String]

  def findById(folderId: String, depth: Int = Int.MaxValue): Option[Folder]

  def findViewableFoldersById(folderId: String, depth: Int = Int.MaxValue, enforcePermission: Boolean = true): Option[Folder]

  def findByPath(titlePath: String, depth: Int = Int.MaxValue): Folder

  def listViewableFolders(parentId: String, page: Page, decorateWithPermissions: Boolean, enforcePermission: Boolean = true): Seq[Folder]

  def registerPersistenceInterceptor(persistenceInterceptor: PersistenceInterceptor[Folder]): Unit

  def checkFolderExists(folderId: String): Unit = {
    if (!exists(folderId)) {
      throw new NotFoundException(s"Folder '$folderId' does not exist in the repository")
    }
  }

  def checkNameIsUnique(parentId: String, name: String): Unit = {
    if (findSubFolderByTitle(parentId, name).isDefined) {
      throw new ItemConflictException(s"Folder with title '$name' already exists under folder by ID '$parentId'")
    }
  }

  protected def paginate(folders: Seq[Folder], page: Page): Seq[Folder] = {
    folders.slice(page.offset.toInt, page.offset.toInt + page.resultsPerPage.toInt)
  }

  protected def decorateWithEffectiveSecurity(folders: Seq[Folder], cache: DecoratorsCache): Seq[Folder] = {
    folders.map(folder => {
      decoratorService.decorate(folder, Seq(EFFECTIVE_SECURITY).asJava, cache)
      folder.setChildren(decorateWithEffectiveSecurity(folder.getChildren.asScala.toSeq, cache).toSet.asJava)
      folder
    })
  }
}

object FolderRepository {
  val PROPERTY_STATUS = "status"
  val PROPERTY_ENABLED = "enabled"

  val ACTIVE_STATUSES_SET: Set[String] = ACTIVE_STATUSES.map(_.name()).toSet
}

class FoldersStoreException(msg: String, cause: Throwable) extends RuntimeException(msg, cause) {
  def this(msg: String) = this(msg, null)
}
