package com.xebialabs.xlrelease.config

import com.codahale.metrics.{InstrumentedExecutorService, InstrumentedScheduledExecutorService}
import com.typesafe.config.ConfigRenderOptions._
import com.typesafe.config.{Config, ConfigException, ConfigFactory, ConfigRenderOptions}
import com.xebialabs.deployit.security.SecretKeyHolder
import com.xebialabs.xlplatform.cluster.ClusterMode.Full
import com.xebialabs.xlplatform.cluster._
import com.xebialabs.xlplatform.config.ConfigLoader
import com.xebialabs.xlplatform.security.validator.regex.RegexValidator.RegexValidation
import com.xebialabs.xlplatform.settings.XlPlatformSettings
import com.xebialabs.xlrelease.config.XlrConfig.DatabaseConfiguration
import com.xebialabs.xlrelease.metrics.XlrMetricRegistry
import com.xebialabs.xlrelease.script.jython.JythonSandboxConfiguration
import com.xebialabs.xlrelease.server.jetty.http.{SameSite, SameSiteHelper}
import com.xebialabs.xlrelease.utils.PrefixedThreadFactory
import grizzled.slf4j.Logging

import java.util.concurrent.{ScheduledThreadPoolExecutor, _}
import java.util.{Properties, List => JList}
import java.{lang, util}
import scala.beans.BeanProperty
import scala.concurrent.duration._
import scala.concurrent.{ExecutionContext, ExecutionContextExecutorService}
import scala.jdk.CollectionConverters._
import scala.language.postfixOps

object XlrConfig {

  var defaultConfigVar: Option[XlrConfig] = None
  var defaultConfigName: String = {
    val overridden: String = System.getProperty("xl.release.config")
    if (overridden == null || overridden.isEmpty) {
      "xl-release.conf"
    } else {
      overridden
    }
  }

  def defaultConfig: XlrConfig = {
    defaultConfigVar match {
      case None =>
        defaultConfigVar = Some(apply(defaultConfigName))
        defaultConfigVar.get
      case Some(config) => config
    }
  }

  lazy val bootConfig: XlrConfig = bootConfig(defaultConfigName)

  def bootConfig(configName: String): XlrConfig = new XlrConfig(configName)

  def apply(configName: String): XlrConfig = {
    val config = new XlrConfig(configName, SecretKeyHolder.get())
    // https://www7.v1host.com/V1Production/defect.mvc/Summary?oidToken=Defect%3A1940958
    // initialisation lazyness is making (some) thread pool initializations to happen when triggered by a script task
    // execution, making use of the wrong (script) AcessControlContext and providing a breeding ground for bugs.
    config.init()
    config
  }

  def getInstance: XlrConfig = defaultConfig

  def reloadConfig(configName: String = defaultConfigName): Unit = {
    defaultConfigVar = Some(apply(configName))
    defaultConfigName = configName
  }

  class DatabaseConfiguration(db: String, rootConfig: Config) {
    lazy val config: Config = rootConfig.getConfig(s"xl.$db")
    lazy val dbDriverClassname: String = sysGetString(s"db-driver-classname", config)
    lazy val dbUrl: String = sysGetString(s"db-url", config)
    lazy val dbUsername: String = sysGetString(s"db-username", config)
    lazy val dbPassword: String = sysGetString(s"db-password", config)
    lazy val poolName: String = sysGetString(s"pool-name", config)
    lazy val maxPoolSize: Int = sysGetInt(s"max-pool-size", config)
    lazy val maxLifeTime: Long = sysGetDuration(s"max-life-time", config)
    lazy val idleTimeout: Long = sysGetDuration(s"idle-timeout", config)
    lazy val minimumIdle: Int = sysGetInt(s"minimum-idle", config)
    lazy val connectionTimeout: Long = sysGetDuration(s"connection-timeout", config)
    lazy val leakConnectionThreshold: Long = sysGetDuration(s"leak-connection-threshold", config)
  }

  private def sysGetString(path: String, config: Config): String = sysGet(path, config)(identity, _.getString)

  private def sysGetInt(path: String, config: Config): Int = sysGet(path, config)(Integer.parseInt, _.getInt)

  private def sysGetBoolean(path: String, config: Config): Boolean = sysGet(path, config)(toBoolean, _.getBoolean)

  private def sysGetDuration(path: String, config: Config, timeUnit: TimeUnit = TimeUnit.MILLISECONDS): Long =
    sysGet(path, config)(Integer.parseInt(_).toLong, c => c.getDuration(_, timeUnit))

  private def sysProps = System.getProperties

  private def sysGet[A](path: String, config: Config)(fromSys: String => A, fromConfig: Config => String => A): A =
    if (sysProps.containsKey(path)) {
      fromSys(sysProps.getProperty(path))
    } else {
      fromConfig(config).apply(path)
    }

  private def toBoolean(s: String): Boolean = s match {
    case "true" | "True" | "1" | "on" => true
    case _ => false
  }

  implicit class ExtendedConfig(config: Config) {
    def getOptionalConfig(configName: String): Option[Config] = if (config.hasPath(configName)) Some(config.getConfig(configName)) else None

    def getOptionalString(path: String): Option[String] = if (config.hasPath(path)) Some(config.getString(path)) else None

    def getOptionalInt(path: String): Option[Integer] = if (config.hasPath(path)) Some(config.getInt(path)) else None

    def getInt(path: String, default: Int): Int = if (config.hasPath(path)) config.getInt(path) else default

    def getString(path: String, default: String): String = if (config.hasPath(path)) config.getString(path) else default

    def getBoolean(path: String, default: Boolean): Boolean = if (config.hasPath(path)) config.getBoolean(path) else default

    def toProperties: Properties = {
      val props = new Properties()
      config.entrySet().stream().forEach(entry => props.put(entry.getKey, entry.getValue.unwrapped()))
      props
    }
  }

}


//noinspection ScalaStyle
class XlrConfig(@BeanProperty val rootConfig: Config) extends Logging {
  import com.xebialabs.xlrelease.config.XlrConfig.ExtendedConfig

  def this(configName: String, secretKeyHolder: SecretKeyHolder) = {
    this(ConfigFactory.defaultOverrides().withFallback(ConfigLoader.loadSecuredWithDynamic(configName, secretKeyHolder)))
  }

  private[config] def this(configName: String) =
    this(ConfigFactory.defaultOverrides().withFallback(ConfigLoader.loadWithDynamic(configName)))

  private def sysGetString(path: String): String = XlrConfig.sysGetString(path, rootConfig)

  private def sysGetBoolean(path: String): Boolean = XlrConfig.sysGetBoolean(path, rootConfig)

  @BeanProperty
  lazy val xl: Config = rootConfig.getConfig("xl")

  if (logger.isDebugEnabled) {
    val configRenderOptions: ConfigRenderOptions = defaults.setOriginComments(false).setJson(false)
    logger.debug(xl.root().render(configRenderOptions))
  }

  def init(): Unit = executors.auxiliaryExecutor.pool

  object executors {

    object scheduler {
      // gives value of xl.executors.scheduler.maxThreadsCount setting, which is there for backward
      // compatibility reasons
      lazy val maxThreadsCount: Int = xl.getInt("executors.scheduler.maxThreadsCount")

      // otherwise, this `scheduler` object is only here for backward compatibility reasons
      // so that native plugins that are using this scheduler as a thread pool or as an
      // execution context (if any) still work. It is wired to miscellaneousExecutor
      lazy val threadPool: ScheduledExecutorService = auxiliaryExecutor.pool
      lazy val executionContext: ExecutionContextExecutorService = ExecutionContext.fromExecutorService(threadPool)
    }

    object releaseTrigger {
      lazy val maxThreadsCount: Int = xl.getInt("executors.releaseTrigger.maxThreadsCount")
      lazy val pool: ExecutorService = {
        val executorService = new ScheduledThreadPoolExecutor(maxThreadsCount, PrefixedThreadFactory("release-trigger"))
        executorService.setRemoveOnCancelPolicy(true)
        if (metrics.enabled) {
          new InstrumentedExecutorService(executorService, XlrMetricRegistry.metricRegistry, "releaseTrigger")
        } else {
          executorService
        }
      }

      lazy val executionContext: ExecutionContextExecutorService = ExecutionContext.fromExecutorService(pool)
    }

    object timeoutExecutor {
      lazy val maxThreadsCount: Int = xl.getInt("executors.timeoutExecutor.maxThreadsCount", 1)
      lazy val pool: ScheduledExecutorService = {
        val executorService = new ScheduledThreadPoolExecutor(maxThreadsCount, PrefixedThreadFactory("backpressure-timeout-executor"))
        executorService.setRemoveOnCancelPolicy(true)
        if (metrics.enabled) {
          new InstrumentedScheduledExecutorService(executorService, XlrMetricRegistry.metricRegistry, "backpressureTimeoutExecutor")
        } else {
          executorService
        }
      }

      lazy val executionContext: ExecutionContextExecutorService = ExecutionContext.fromExecutorService(pool)

    }

    object springScheduler {
      lazy val maxThreadsCount: Int = xl.getInt("executors.springScheduler.maxThreadsCount")
    }

    object pollingExecutor {
      lazy val maxThreadsCount: Int = xl.getInt("executors.polling.maxThreadsCount")
      lazy val pool: ScheduledExecutorService = {
        val scheduledExecutorService = new ScheduledThreadPoolExecutor(maxThreadsCount, PrefixedThreadFactory("polling"))
        scheduledExecutorService.setRemoveOnCancelPolicy(true)
        if (metrics.enabled) {
          new InstrumentedScheduledExecutorService(scheduledExecutorService, XlrMetricRegistry.metricRegistry, "polling")
        } else {
          scheduledExecutorService
        }
      }
    }

    object preArchivingExecutor {
      lazy val maxThreadsCount: Int = xl.getInt("executors.preArchiving.maxThreadsCount")
    }

    object auxiliaryExecutor {
      lazy val maxThreadsCount: Int = xl.getInt("executors.auxiliary.maxThreadsCount")
      lazy val pool: ScheduledExecutorService = {
        val scheduledExecutorService = new ScheduledThreadPoolExecutor(maxThreadsCount, PrefixedThreadFactory("auxiliary"))
        scheduledExecutorService.setRemoveOnCancelPolicy(true)
        if (metrics.enabled) {
          new InstrumentedScheduledExecutorService(scheduledExecutorService, XlrMetricRegistry.metricRegistry, "auxiliary")
        } else {
          scheduledExecutorService
        }
      }
      lazy val executionContext: ExecutionContextExecutorService = ExecutionContext.fromExecutorService(pool)
    }
  }

  object license {
    lazy val location: String = xl.getString("license.location")
  }

  object confFile {
    lazy val location: String = xl.getString("conf-file.location")
  }

  object quartz {
    lazy val timezone: String = xl.getString("quartz.timezone")
  }

  object timeouts {
    val CUSTOM_SCRIPT_TASK_TIMEOUT_KEY = "customScriptTaskTimeout"
    val SCRIPT_TASK_TIMEOUT_KEY = "scriptTaskTimeout"
    val FAILURE_HANDLER_TIMEOUT_KEY = "failureHandlerTimeout"
    val PRECONDITION_TIMEOUT_KEY = "preconditionTimeout"
    val SCRIPT_TASK_TIMEOUT_ENABLED_KEY = SCRIPT_TASK_TIMEOUT_KEY + "Enabled"
    val CUSTOM_SCRIPT_TASK_TIMEOUT_ENABLED_KEY = CUSTOM_SCRIPT_TASK_TIMEOUT_KEY + "Enabled"
    val PRECONDITION_TIMEOUT_ENABLED_KEY = PRECONDITION_TIMEOUT_KEY + "Enabled"

    private lazy val timeouts = xl.getConfig("timeouts")
    lazy val systemInitialization: FiniteDuration = timeouts.getDuration("systemInitialization", TimeUnit.SECONDS) seconds
    lazy val systemTermination: FiniteDuration = timeouts.getDuration("systemTermination", TimeUnit.SECONDS) seconds
    lazy val taskSchedulerGraceShutdownPeriod: FiniteDuration = timeouts.getDuration("taskSchedulerGraceShutdownPeriod", TimeUnit.SECONDS) seconds
    lazy val releaseActionResponse: FiniteDuration = timeouts.getDuration("releaseActionResponse", TimeUnit.SECONDS) seconds
    lazy val releaseActorReceive: FiniteDuration = timeouts.getDuration("releaseActorReceive", TimeUnit.MILLISECONDS) millis
    lazy val ddataConsistencyTimeout: FiniteDuration = timeouts.getDuration("ddataConsistencyTimeout", TimeUnit.MILLISECONDS) millis
    lazy val scmConnectorRequestTimeout: FiniteDuration = timeouts.getDuration("scmConnectorRequestTimeout", TimeUnit.SECONDS) seconds
    lazy val messageRetryTimeout: FiniteDuration = timeouts.getDuration("messageRetryTimeout", TimeUnit.SECONDS) seconds
    lazy val messageRetryAttempts: Int = timeouts.getInt("messageRetryAttempts")
    lazy val nodeFailureDetectionPeriod: FiniteDuration = timeouts.getDuration("nodeFailureDetectionPeriod", TimeUnit.MILLISECONDS) millis
    lazy val staleReservedJobDetectionPeriod: FiniteDuration = timeouts.getDuration("staleReservedJobPeriod", TimeUnit.MILLISECONDS) millis

    lazy val failureHandlerTimeout: FiniteDuration = timeouts.getDuration("failureHandlerTimeout", TimeUnit.MILLISECONDS) millis
    lazy val scriptTaskTimeoutEnabled: Boolean = timeouts.getBoolean(SCRIPT_TASK_TIMEOUT_ENABLED_KEY)
    lazy val scriptTaskTimeout: FiniteDuration = timeouts.getDuration(SCRIPT_TASK_TIMEOUT_KEY, TimeUnit.MILLISECONDS) millis
    lazy val customScriptTaskTimeoutEnabled: Boolean = timeouts.getBoolean(CUSTOM_SCRIPT_TASK_TIMEOUT_ENABLED_KEY)
    lazy val customScriptTaskTimeout: FiniteDuration = timeouts.getDuration(CUSTOM_SCRIPT_TASK_TIMEOUT_KEY, TimeUnit.MILLISECONDS) millis
    lazy val preconditionTimeoutEnabled: Boolean = timeouts.getBoolean(PRECONDITION_TIMEOUT_ENABLED_KEY)
    lazy val preconditionTimeout: FiniteDuration = timeouts.getDuration(PRECONDITION_TIMEOUT_KEY, TimeUnit.MILLISECONDS) millis
  }

  def timeoutSettings: timeouts.type = timeouts

  def cluster = ClusterConfig(xl.getConfig("cluster"))

  object sharding {
    lazy val numberOfReleaseShards: Int = xl.getConfig("cluster").getInt("number-of-shards")
  }

  object durations {
    private lazy val durations = xl.getConfig("durations")
    lazy val notifyOverduePlanItemsInterval: FiniteDuration = durations.getDuration("notifyOverduePlanItemsInterval", TimeUnit.MILLISECONDS) millis
    lazy val notifyDueSoonTasksInterval: FiniteDuration = durations.getDuration("notifyDueSoonTasksInterval", TimeUnit.MILLISECONDS) millis
    lazy val customScriptTaskScheduleInterval: FiniteDuration = {
      val interval = durations.getDuration("customScriptTaskScheduleInterval", TimeUnit.MILLISECONDS).millis
      if (interval < 1.second) {
        throw new IllegalArgumentException(
          s"Value for customScriptTaskScheduleInterval cannot be less than 1 second but was $interval")
      }
      interval
    }
    lazy val scriptOutputPollingInterval: FiniteDuration = durations.getDuration("scriptOutputPollingInterval", TimeUnit.MILLISECONDS) millis
    lazy val newWatcherPublishDelay: FiniteDuration = durations.getDuration("newWatcherPublishDelay", TimeUnit.MILLISECONDS) millis
    lazy val watcherUnregisterDelay: FiniteDuration = durations.getDuration("watcherUnregisterDelay", TimeUnit.MILLISECONDS) millis
    lazy val releaseActivationDelay: FiniteDuration = durations.getDuration("releaseActivationDelay", TimeUnit.MILLISECONDS) millis
    lazy val triggerDataPurgeJobInterval: FiniteDuration = durations.getDuration("triggerDataPurgeJobInterval", TimeUnit.MINUTES) minute
    lazy val jobQueuePopulateInterval: FiniteDuration = durations.getDuration("jobQueuePopulateInterval", TimeUnit.MILLISECONDS) millis
  }

  def risk: RiskSettings.type = RiskSettings

  object RiskSettings {
    private lazy val risk = xl.getConfig("features.risk")
    lazy val calculationEnabled: Boolean = risk.getBoolean("calculation.enabled", true)
  }

  object akka {
    lazy val akkaPersistenceConfig: Config = xl.getConfig("akka-persistence")
    lazy val kryoSerializationConfig: Config = xl.withOnlyPath("akka-kryo-serialization")
    lazy val nonClusteredConfig: Config = xl.withOnlyPath("akka").withFallback(kryoSerializationConfig).withFallback(akkaPersistenceConfig)
    lazy val clusteredConfig: Config = cluster.config.withOnlyPath("akka").withFallback(nonClusteredConfig)

    lazy val config: Config = {
      val modeConfig = if (clusterMode == Full) clusteredConfig else nonClusteredConfig
      modeConfig.withFallback(rootConfig.withOnlyPath("xl.dispatchers"))
    }

    lazy val systemName: String = if (clusterMode == Full) cluster.name else xl.getString("systemName")
  }

  object development {
    private lazy val development = xl.getConfig("development")
    lazy val restSlowDownDelay: FiniteDuration = development.getDuration("restSlowDownDelay", TimeUnit.MILLISECONDS) millis
    lazy val keepContainer: Boolean = development.getBoolean("keepContainer", false)
  }

  object api {
    private lazy val api = xl.getConfig("api")
    lazy val corsEnabled: Boolean = api.getBoolean("corsEnabled")
  }

  object maintenance{
    private lazy val api = xl.getConfig("maintenance")
    lazy val maintenanceModeEnabled: Boolean = api.getBoolean("enabled")
    lazy val isApiAccessRestricted: Boolean = api.getBoolean("restrictApiAccess")
  }

  def server: ServerSettings.type = ServerSettings

  object ServerSettings {
    private lazy val server = xl.getConfig("server")

    object http {
      private lazy val http = server.getConfig("http")

      object gzip {
        private lazy val gzip = http.getConfig("gzip")
        lazy val enabled: Boolean = gzip.getBoolean("enabled")
        lazy val minSize: lang.Long = gzip.getBytes("minSize")
        lazy val compression: Int = gzip.getInt("compression")
        lazy val excludedPaths: util.List[String] = gzip.getStringList("excludedPaths")
      }

      object cache {
        private lazy val cache = http.getConfig("cache")
        lazy val enabled: Boolean = cache.getBoolean("enabled")
      }

      object csrf {
        private lazy val csrf = http.getConfig("csrf")
        lazy val enabled: Boolean = csrf.getBoolean("enabled")
      }

      object cookie {
        private lazy val cookie = http.getConfig("cookie")
        lazy val sameSite: SameSite = SameSiteHelper.getSameSiteValue(cookie.getString("sameSite", "Lax"))
      }
    }

    def upload: UploadSettings.type = UploadSettings

    object UploadSettings {
      private lazy val config = server.getConfig("upload")

      lazy val maxSize: Int = config.getInt("max-size", 100)
      lazy val maxSizeBytes: Int = maxSize * 1024 * 1024
      lazy val shouldAnalyzeContent: Boolean = config.getBoolean("analyze-content", false)

      lazy val allowedFileTypes: Map[String, List[String]] = if (config.hasPathOrNull("allowed-content-types")) {
        if (config.getIsNull("allowed-content-types")) {
          Map.empty
        } else {
          config.getObject("allowed-content-types").asScala.map {
            case (fileType, filePatterns) => (fileType, filePatterns.unwrapped().asInstanceOf[JList[String]].asScala.toList)
          }.toMap
        }
      } else {
        Map("*" -> List("*"))
      }
    }

    def session: SessionSettings.type = SessionSettings

    object SessionSettings {
      private lazy val session = server.getConfig("session")

      lazy val maximumSessions: Int = session.getInt("maximumSessions", -1)
      lazy val exceptionIfMaximumExceeded: Boolean = session.getBoolean("exceptionIfMaximumExceeded", false)

      object storage {
        private lazy val storage = session.getConfig("storage")
        lazy val enabled: Boolean = storage.getBoolean("enabled")
        lazy val cleanupCron: String = storage.getString("cleanup-cron")
      }

      object rememberMeSettings {
        private lazy val rememberMe = session.getConfig("remember-me")
        lazy val key: String = rememberMe.getString("key")
        lazy val tokenValidity: Int = rememberMe.getDuration("tokenValidity", TimeUnit.SECONDS).toInt
      }

    }
  }

  object initialization {
    private lazy val initialization = xl.getConfig("initialization")
    lazy val createSampleTemplates: Boolean = initialization.getBoolean("createSampleTemplates")
  }

  object security extends SecuritySettings with ScriptingSecurity with SecretComplexity with AcceptEncryptedSecretsSecurity

  trait SecuritySettings {
    lazy val security: Config = xl.getConfig("security")

    lazy val isOidcEnabled: Boolean = if (security.hasPath("auth.providers.oidc.enabled")) {
      security.getBoolean("auth.providers.oidc.enabled")
    } else if (security.hasPath("auth.providers.oidc")){
      true
    } else {
      false
    }

    private lazy val additionalParametersConfigKey = "auth.providers.oidc.additionalParameters"

    lazy val oidcAdditionalParameters: util.Map[String, Object] = if (security.hasPathOrNull(additionalParametersConfigKey) && !security.getIsNull(additionalParametersConfigKey)) {
      security.getObject(additionalParametersConfigKey).unwrapped()
    } else {
      new util.HashMap[String, Object]()
    }

  }

  trait ScriptingSecurity {
    def security: Config

    object scripting {
      private lazy val scripting = security.getConfig("scripting")

      object sandbox {
        private lazy val sandbox = scripting.getConfig("sandbox")

        lazy val decryptPasswords: Boolean = if (xl.hasPath("features.script.sandbox.decryptPasswords")) { // legacy path
          xl.getBoolean("features.script.sandbox.decryptPasswords")
        } else {
          sandbox.getBoolean("decryptPasswords")
        }

        lazy val isEnabled: Boolean = if (xl.hasPath("features.script.sandbox.enabled")) { // legacy path
          xl.getBoolean("features.script.sandbox.enabled")
        } else {
          sandbox.getBoolean("enabled")
        }

        lazy val autoRefreshScriptPolicy = sandbox.getBoolean("autoRefreshScriptPolicy")

        object jython {
          private lazy val jython = sandbox.getConfig("jython")

          lazy val restrictedModules: util.List[String] = jython.getStringList("restricted-modules")
          lazy val restrictedFunctions: util.List[String] = jython.getStringList("restricted-functions")
          lazy val restrictedAttributes: util.List[String] = jython.getStringList("restricted-attributes")

          lazy val isolatePlugins: Boolean = jython.getBoolean("isolate-plugins")
        }

      }

    }

  }

  trait SecretComplexity {
    def security: Config

    object secret {
      private lazy val secretComplexity = security.getConfig("secret-complexity")
      private val validations: JList[String] = secretComplexity.getStringList("regex-validations")
      lazy val regexValidations: List[RegexValidation] = validations.asScala.toList.grouped(2)
        .flatMap {
          case List(regex, message) => Some(RegexValidation(regex.r, message))
          case _ =>
            logger.error("One of the patterns in secret-complexity.regex-validations was not properly formatted.")
            None
        }
        .toList
    }

  }

  trait AcceptEncryptedSecretsSecurity {
    def security: Config

    object acceptEncryptedSecrets {
      private lazy val acceptEncrypted = security.getConfig("accept-encrypted-secrets")
      lazy val enabled: Boolean = acceptEncrypted.getBoolean("enabled", false)
    }

  }

  object features extends FeaturesSettings

  trait FeaturesSettings {
    private lazy val features = xl.getConfig("features")

    object releaseOwner {
      lazy val releaseOwnerConfig: Config = features.getConfig("release-owner")

      object taskTransition {
        lazy val taskTransitionConfig: Config = releaseOwnerConfig.getConfig("task-transition")
        lazy val taskTransitionAllowed: Boolean = taskTransitionConfig.getBoolean("allowed")
      }
    }

    object plugins {
      lazy val config: Config = features.getConfig("plugins")
    }

    object releaseActors {
      lazy val config: Config = features.getConfig("release-actors")

      lazy val releaseCacheEnabled: Boolean = config.getBoolean("cache.enabled")
    }

    object userLastActiveActor {
      lazy val config: Config = features.getConfig("user-last-active-actor")

      lazy val maxBufferSize: Int = config.getInt("max-buffer-size")

      lazy val batchSize: Int = config.getInt("batch-size")

      lazy val flushInterval: FiniteDuration = config.getDuration("flush-interval", TimeUnit.MILLISECONDS) millis

      require(batchSize > 0, "xl.features.user-last-active-actor.batch-size must be a positive integer")
      require(maxBufferSize > 0, "xl.features.user-last-active-actor.max-buffer-size must be a positive integer")
      require(flushInterval.toSeconds >= 1, "xl.features.user-last-active-actor.flush-interval should be at least 1 second")
    }

    object updates {
      lazy val config: Config = features.getConfig("updates")

      // REL-7726 and REL-7258 - parallel updates can cause a lock
      lazy val parallel: Boolean = config.getBoolean("parallel", true)
    }

    object notifications {
      lazy val config: Config = features.getConfig("notifications")

      lazy val dueSoonEnabled: Boolean = config.getBoolean("due-soon.enabled", true)

      lazy val overdueEnabled: Boolean = config.getBoolean("overdue.enabled", true)

      lazy val notifyInactiveUsers: Boolean = config.getBoolean("notify-inactive-users.enabled", false)
    }

    object gateActor {
      lazy val config: Config = features.getConfig("gate-actor")

      lazy val enabled: Boolean = config.getBoolean("enabled", true)

      lazy val graceDuration: FiniteDuration = config.getDuration("graceDuration", TimeUnit.MILLISECONDS) millis

      lazy val unstuckInterval: FiniteDuration = config.getDuration("unstuckInterval", TimeUnit.MILLISECONDS) millis
    }

    @Deprecated(forRemoval = true)
    object lockTask {
      lazy val config: Config = features.getConfig("lock-task")

      lazy val assignmentEnabled: Boolean = config.getBoolean("assignment.enabled", true)
    }

    object webhooks {
      private lazy val webhooksConfig: Config = features.getConfig("webhooks")

      object retentionPolicy {
        private lazy val config: Config = webhooksConfig.getConfig("retention-policy")

        lazy val schedule: String = config.getString("schedule")
      }

    }

    object serialization {
      private lazy val serializationConfig: Config = features.getConfig("serialization")
      lazy val failOnUnresolvableReferences: Boolean = serializationConfig.getBoolean("failOnUnresolvableReferences")
    }

    object analytics {
      private lazy val analyticsConfig: Config = features.getConfig("analytics")
      lazy val pendoQueueSize: Int = analyticsConfig.getInt("pendoQueueSize")
      lazy val pendoMetricsCron: String = analyticsConfig.getString("pendoMetricsCron", "0 0 3 * * *")
    }

    object liveDbCredentialUpdate {
      lazy val liveDbConfig: Config = features.getConfig("liveDbCredentialUpdate")
      lazy val enabled: Boolean = liveDbConfig.getBoolean("enabled", false)
    }
  }


  trait ReportEngineConfiguration {
    def config: Config

    private lazy val engineConfig: Config = config.getConfig("engine")

    lazy val engine: EngineConfig.type = EngineConfig

    object EngineConfig {
      lazy val maxThreadsCount: Int = engineConfig.getInt("maxThreadsCount", 10)
      lazy val location: String = engineConfig.getString("location", "reports")
      lazy val cleanUpInterval: Long = engineConfig.getDuration("cleanUpInterval", TimeUnit.MILLISECONDS)
    }

  }

  object database extends DatabaseConfiguration("database", rootConfig)

  object reporting extends DatabaseConfiguration("reporting", rootConfig) with ReportEngineConfiguration

  def metrics: MetricsSettings.type = MetricsSettings

  object MetricsSettings {
    private lazy val metrics = xl.getConfig("metrics")
    lazy val enabled: Boolean = metrics.getBoolean("enabled")
  }

  def repository: RepositoryConfig.type = RepositoryConfig

  object RepositoryConfig {
    private lazy val repository = xl.getConfig("repository")

    def config: Config = repository

    lazy val workDir: String = {
      try {
        sysGetString("xl.repository.workDir")
      } catch {
        case _: ConfigException.Missing => "work"
      }
    }

    lazy val jobLogDir: String = repository.getString("jobLogDir", "work/job_logs")

    def defaults: DefaultConfig.type = DefaultConfig

    lazy val decryptPasswords: Boolean = repository.getBoolean("decryptPasswords")

    lazy val sql: Boolean = sysGetBoolean("xl.repository.sql")

    object DefaultConfig {
      private lazy val default = xl.getConfig("default")

      def config: Config = default
    }

  }

  object upgrader {
    private lazy val upgrader = xl.getConfig("upgrader")
    lazy val forceRemoveMissingTypes: Boolean = upgrader.getBoolean("forceRemoveMissingTypes")
  }

  // accessors for Java/spring context files

  def triggerThreadPool: ExecutorService = this.executors.releaseTrigger.pool

  def timeoutExecutorThreadPool: ScheduledExecutorService = this.executors.timeoutExecutor.pool

  def pollingExecutor: ScheduledExecutorService = this.executors.pollingExecutor.pool

  def auxiliaryExecutor: ScheduledExecutorService = this.executors.auxiliaryExecutor.pool

  def xlrRepositoryJdbcUrl: String = this.database.dbUrl

  def xlrRepositoryDriver: String = this.database.dbDriverClassname

  def xlrRepositoryUsername: String = this.database.dbUsername

  def xlrRepositoryPassword: String = this.database.dbPassword

  def reportingDriver: String = this.reporting.dbDriverClassname

  def reportingUrl: String = this.reporting.dbUrl

  def reportingUsername: String = this.reporting.dbUsername

  def reportingPassword: String = this.reporting.dbPassword

  def reportingMaxPoolSize: Int = this.reporting.maxPoolSize

  def reportingPoolName: String = this.reporting.poolName

  def reportingMaxLifetime: Long = this.reporting.maxLifeTime

  def reportingIdleTimeout: Long = this.reporting.idleTimeout

  def reportingMinimumIdle: Int = this.reporting.minimumIdle

  def reportingConnectionTimeout: Long = this.reporting.connectionTimeout

  def reportingLeakConnectionThreshold: Long = this.reporting.leakConnectionThreshold

  def reportingEngineLocation: String = this.reporting.engine.location

  def serverExtension_rootPath: String = XlPlatformSettings.apply(rootConfig).ServerExtension.rootPath

  def confFile_location: String = this.confFile.location

  def license_location: String = this.license.location

  def isClusterEnabled: Boolean = cluster.mode != ClusterMode.Standalone

  def clusterMode: ClusterMode = cluster.mode

  def isClusterWithSeedNodes: Boolean = cluster.config.hasPath("akka.cluster.seed-nodes")

  def isActiveOnStartup: Boolean = cluster.mode == ClusterMode.Standalone

  def systemTerminationSeconds: Long = this.timeouts.systemTermination.toSeconds

  def taskSchedulerGraceShutdownPeriodSeconds: Long = this.timeouts.taskSchedulerGraceShutdownPeriod.toSeconds

  def taskSchedulerGraceShutdownPeriod: Long = this.timeouts.taskSchedulerGraceShutdownPeriod.toSeconds

  def development_restSlowDownDelay: FiniteDuration = this.development.restSlowDownDelay

  def api_corsEnabled: Boolean = this.api.corsEnabled

  def maintenanceModeEnabled: Boolean = this.maintenance.maintenanceModeEnabled

  def maintenanceModeRestrictApiAccess: Boolean = this.maintenance.isApiAccessRestricted

  def server_http_gzip_enabled: Boolean = this.ServerSettings.http.gzip.enabled

  def server_http_gzip_minSize: lang.Long = this.ServerSettings.http.gzip.minSize

  def server_http_gzip_compression: Int = this.ServerSettings.http.gzip.compression

  def server_http_gzip_excludedPaths: util.List[String] = this.ServerSettings.http.gzip.excludedPaths

  def server_http_cache_enabled: Boolean = this.ServerSettings.http.cache.enabled

  def server_http_csrf_enabled: Boolean = this.ServerSettings.http.csrf.enabled

  def server_http_cookie_sameSite: SameSite = this.ServerSettings.http.cookie.sameSite

  def server_session_maximum_sessions: Int = this.ServerSettings.session.maximumSessions

  def server_session_exception_if_maximum_exceeded: Boolean = this.ServerSettings.session.exceptionIfMaximumExceeded

  def server_session_remember_me_key: String = this.ServerSettings.session.rememberMeSettings.key

  def server_session_remember_me_token_validity_seconds: Int = this.ServerSettings.session.rememberMeSettings.tokenValidity

  def server_session_storage_enabled: Boolean = this.ServerSettings.session.storage.enabled

  def server_session_storage_cleanupCron: String = this.ServerSettings.session.storage.cleanupCron

  def initialization_createSampleTemplates: Boolean = this.initialization.createSampleTemplates

  def durations_customScriptTaskScheduleInterval: FiniteDuration = this.durations.customScriptTaskScheduleInterval

  def durations_scriptOutputPollingInterval: FiniteDuration = this.durations.scriptOutputPollingInterval

  def repository_decryptPasswords: Boolean = this.repository.decryptPasswords

  def pluginsConfig: Config = this.features.plugins.config

  def isRiskCalculationEnabled: Boolean = this.risk.calculationEnabled

  def isScriptSandboxEnabled: Boolean = this.security.scripting.sandbox.isEnabled

  def isNotifyInactiveUsers: Boolean = this.features.notifications.notifyInactiveUsers

  @Deprecated
  def isScriptSandboxDecryptPasswords: Boolean = this.security.scripting.sandbox.decryptPasswords

  def isAutoRefreshScriptPolicy: Boolean = this.security.scripting.sandbox.autoRefreshScriptPolicy

  def regexPasswordValidationList: List[RegexValidation] = this.security.secret.regexValidations

  def isOidcEnabled: Boolean = this.security.isOidcEnabled

  def oidcAdditionalParameters(): util.Map[String, Object] = this.security.oidcAdditionalParameters

  @Deprecated(forRemoval = true)
  def isLockTaskAssignmentDisabled: Boolean = !this.features.lockTask.assignmentEnabled

  def jythonSandboxConfiguration(): JythonSandboxConfiguration = {
    val jythonScripting = this.security.scripting.sandbox.jython
    JythonSandboxConfiguration(
      jythonScripting.restrictedModules,
      jythonScripting.restrictedFunctions,
      jythonScripting.restrictedAttributes
    )
  }

  def isReleaseOwnerTaskTransitionAllowed: Boolean = this.features.releaseOwner.taskTransition.taskTransitionAllowed

  def isLiveDbCredentialUpdateEnabled: Boolean = this.features.liveDbCredentialUpdate.enabled

}
