package com.xebialabs.xlrelease.actors

import akka.actor._
import akka.pattern.{GracefulStopSupport, pipe}
import akka.util.Timeout
import com.xebialabs.xlrelease.actors.ReleaseExecutionActorMessages.Command
import com.xebialabs.xlrelease.actors.ReleaseSupervisorActor._
import com.xebialabs.xlrelease.actors.extension.AddressExtension
import com.xebialabs.xlrelease.actors.sharding.ReleaseShardingMessages.{PrepareForBalance, ReleaseAction, ReleaseExecutorActorTerminated}
import com.xebialabs.xlrelease.config.XlrConfig

import scala.concurrent.ExecutionContext

object ReleaseActor {

  val name = "release-actor"

  def props(actorServiceHolder: ActorServiceHolder, executionActorMaker: ReleaseExecutorActorMaker, releaseId: String, isTemplate: Boolean, xlrConfig: XlrConfig): Props = Props(
    new ReleaseActor(actorServiceHolder, executionActorMaker, releaseId, isTemplate, xlrConfig)
  ).withDispatcher("xl.dispatchers.release-dispatcher")

  /**
   * This message should be ignored by [[ReleaseActor]]. It is needed to activate the actor after sharding rebalancing.
   */
  case object Activate

}

class ReleaseActor(actorServiceHolder: ActorServiceHolder,
                   executionActorMaker: ReleaseExecutorActorMaker,
                   releaseId: String,
                   isTemplate: Boolean,
                   xlrConfig: XlrConfig) extends Actor with Escalate with Unstash with ActorLogging with GracefulStopSupport {

  private lazy val executorActorRef = executionActorMaker(context, context.parent, releaseId)
  private var schedulerActorRef: Option[ActorRef] = None
  private lazy val askTimeout: Timeout = xlrConfig.timeouts.releaseActionResponse

  override def preStart(): Unit = {
    super.preStart()
    log.debug(s"Starting ReleaseActor for release '$releaseId'")
  }

  override def postStop(): Unit = {
    super.postStop()
    ReleaseActorCache.remove(releaseId)
    log.debug(s"Stopping release actor tree for $releaseId at ${AddressExtension(context.system).address}.")
  }

  override def receive: PartialFunction[Any, Unit] = executeCommand orElse garbageCollect orElse handleRebalancing orElse handleShutdown

  def handleShutdown: Receive = {
    case GracefulPoisonPill =>
      implicit val ec: ExecutionContext = context.dispatcher
      log.debug(s"Going to gracefully stop ReleaseExecutionActor for ${releaseId}")
      gracefulStop(executorActorRef, xlrConfig.timeouts.releaseActionResponse).recover(e => {
        log.error(s"Unable to stop ReleaseExecutionActor for ${releaseId} within timeout", e)
        true
      }).andThen {
        case _ => context.stop(self)
      }
  }

  def executeCommand: Receive = {
    case cmd: Command =>
      log.debug(s"${if (isTemplate) "Template" else "Release"} '${releaseId}' executing command: $cmd")
      executorActorRef forward cmd
    case action: ReleaseAction =>
      val parent = context.parent
      log.debug(s"${if (isTemplate) "Template" else "Release"} '${releaseId}' forwarding command: $action to parent '$parent'")
      parent forward action
  }

  def garbageCollect: Receive = {
    case gc@GcRequest if isTemplate =>
      log.debug(s"GC: allowed for template '$releaseId' on $gc")
      sender() ! ReleasePoisonPill
    case msg@GcRequest =>
      executorActorRef forward msg
  }

  def handleRebalancing: Receive = {
    case msg@PrepareForBalance =>
      implicit val ec: ExecutionContext = context.dispatcher
      log.debug(s"Going to gracefully stop execution actor for ${releaseId}")
      gracefulStop(executorActorRef, askTimeout.duration).recover(e => {
        log.error(s"Unable to stop releaseExecutionActor for ${releaseId} within timeout, consider increasing xl.cluster.akka.cluster.sharding.handoff-timeout", e)
        true
      }).map(_ => ReleaseExecutorActorTerminated).pipeTo(self)
    case ReleaseExecutorActorTerminated =>
      log.debug(s"Received ReleaseExecutorActorTerminated for ${releaseId}, going to stop")
      context.stop(context.parent)
    case ReleaseDeleted =>
      context.parent ! ReleasePoisonPill
  }

}
