package com.xebialabs.xlrelease.api.v1;

import java.io.ByteArrayOutputStream;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.stream.Collectors;
import jakarta.ws.rs.core.MediaType;
import org.jboss.resteasy.specimpl.BuiltResponse;
import org.junit.Before;
import org.junit.Rule;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.restdocs.JUnitRestDocumentation;
import org.springframework.restdocs.mockmvc.RestDocumentationResultHandler;
import org.springframework.restdocs.operation.preprocess.OperationPreprocessor;
import org.springframework.restdocs.payload.FieldDescriptor;
import org.springframework.restdocs.payload.JsonFieldType;
import org.springframework.restdocs.payload.RequestFieldsSnippet;
import org.springframework.restdocs.payload.ResponseFieldsSnippet;
import org.springframework.restdocs.request.QueryParametersSnippet;
import org.springframework.test.web.servlet.MockMvc;
import org.springframework.test.web.servlet.MvcResult;
import org.springframework.test.web.servlet.setup.MockMvcBuilders;
import org.springframework.web.context.WebApplicationContext;
import com.fasterxml.jackson.annotation.JsonAutoDetect.Visibility;
import com.fasterxml.jackson.annotation.PropertyAccessor;
import com.fasterxml.jackson.databind.ObjectMapper;

import com.xebialabs.deployit.plugin.api.reflect.PropertyDescriptor;
import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.deployit.plumbing.ConfigurationItemReaderWriter;
import com.xebialabs.deployit.plumbing.ConfigurationItemsReaderWriter;
import com.xebialabs.deployit.plumbing.RequestLocal;
import com.xebialabs.xlrelease.XLReleaseIntegrationTest;
import com.xebialabs.xlrelease.domain.Release;
import com.xebialabs.xlrelease.domain.status.ReleaseStatus;
import com.xebialabs.xlrelease.rules.LoginRule;

import static com.google.common.collect.Lists.newArrayList;
import static com.xebialabs.xlrelease.api.ApiService.PAGE;
import static com.xebialabs.xlrelease.api.ApiService.RESULTS_PER_PAGE;
import static com.xebialabs.xlrelease.domain.status.ReleaseStatus.TEMPLATE;
import static java.lang.String.format;
import static org.springframework.restdocs.mockmvc.MockMvcRestDocumentation.document;
import static org.springframework.restdocs.mockmvc.MockMvcRestDocumentation.documentationConfiguration;
import static org.springframework.restdocs.operation.preprocess.Preprocessors.preprocessRequest;
import static org.springframework.restdocs.operation.preprocess.Preprocessors.preprocessResponse;
import static org.springframework.restdocs.operation.preprocess.Preprocessors.prettyPrint;
import static org.springframework.restdocs.payload.PayloadDocumentation.fieldWithPath;
import static org.springframework.restdocs.payload.PayloadDocumentation.relaxedRequestFields;
import static org.springframework.restdocs.payload.PayloadDocumentation.relaxedResponseFields;
import static org.springframework.restdocs.payload.PayloadDocumentation.responseFields;
import static org.springframework.restdocs.request.RequestDocumentation.parameterWithName;
import static org.springframework.restdocs.request.RequestDocumentation.queryParameters;


public abstract class BaseApiDocumentationTest extends XLReleaseIntegrationTest {

    @Autowired
    private WebApplicationContext webApplicationContext;

    @Autowired
    private ObjectMapper jacksonObjectMapper;

    @Autowired
    private ConfigurationItemReaderWriter configurationItemReaderWriter;

    @Autowired
    private NonEncryptingPasswordItemReaderWriter nonEncryptingPasswordItemReaderWriter;

    @Autowired
    private ConfigurationItemsReaderWriter configurationItemsReaderWriter;

    @Rule
    public JUnitRestDocumentation restDocumentation = new JUnitRestDocumentation("build/generated-snippets");

    @Rule
    public LoginRule loginRule = LoginRule.loginWithRoleAdmin("admin");

    protected MockMvc mockMvc;

    protected RestDocumentationResultHandler documentationHandler;

    @Before
    public void setUp() {
        this.documentationHandler = document("{class-name}/{method-name}",
                preprocessRequest(getRequestDocsPreprocessors()),
                preprocessResponse(prettyPrint()));
        this.mockMvc = MockMvcBuilders.webAppContextSetup(webApplicationContext)
                .apply(documentationConfiguration(this.restDocumentation).uris()
                        .withScheme("http")
                        .withHost("localhost")
                        .withPort(5516))
                .alwaysDo(this.documentationHandler)
                .build();
        RequestLocal.initMap();
    }

    protected OperationPreprocessor[] getRequestDocsPreprocessors() {
        return new OperationPreprocessor[]{prettyPrint()};
    }

    protected void markForDeletion(final MvcResult result) {
        BuiltResponse response = ((BuiltResponse) result.getModelAndView().getModel().get("responseInvoker"));
        ConfigurationItem ci = (ConfigurationItem) response.getEntity();
        markForDeletion(ci.getId());
    }

    protected QueryParametersSnippet paginatedRequestParametersSnippet() {
        return queryParameters(
                parameterWithName(PAGE)
                        .description("The page to fetch, zero-based; default value is 0")
                        .optional(),
                parameterWithName(RESULTS_PER_PAGE)
                        .description("Number of results per page, default value is 100")
                        .optional()
        );
    }

    protected RequestFieldsSnippet ciRequestFieldsSnippet(final Type ciType) {
        return relaxedRequestFields(ciFields(ciType)).and(ignoredFields());
    }

    protected ResponseFieldsSnippet ciResponseFieldsSnippet(final Type ciType) {
        return ciResponseFieldsSnippet(ciType, "");
    }

    protected ResponseFieldsSnippet ciResponseFieldsSnippet(final Type ciType, String prefix) {
        return relaxedResponseFields().andWithPrefix(prefix, ciFields(ciType)).andWithPrefix(prefix, ignoredFields());
    }

    protected RequestFieldsSnippet variableRequestSnippet() {
        return relaxedRequestFields(
                fieldWithPath("key").description("The unique name of the variable in the way it is used in template or release, without curly braces"),
                fieldWithPath("value").description("Value of the release variable or default value of the template variable"),
                fieldWithPath("requiresValue").description("Shows if an empty value is a valid value for this variable"),
                fieldWithPath("showOnReleaseStart").description("Shows if this variable will be shown on create release page")
        );
    }

    protected RequestFieldsSnippet releaseFiltersSnippet() {
        return relaxedRequestFields(
                fieldWithPath("title").type("String").description("Case-insensitive matches the part of the release title"),
                fieldWithPath("tags").type("String").description("Matches the releases containing all of the specified tags"),
                fieldWithPath("anyTags").description("Matches the releases containing any of the specified tags"),
                fieldWithPath("timeFrame").type("String").description("The time-frame to search releases in. " +
                        "Valid values: LAST_MONTH, LAST_THREE_MONTHS, LAST_SIX_MONTHS, LAST_YEAR. " +
                        "Specify RANGE to filter by a custom from-to date range"),
                fieldWithPath("from").type("Long").description("Matches the releases with end date after or equal to this date"),
                fieldWithPath("to").type("Long").description("Matches the releases with start date before this date"),
                fieldWithPath("active").description("Matches the releases with the IN_PROGRESS, FAILED, FAILING or PAUSED status"),
                fieldWithPath("planned").description("Matches the releases with the PLANNED status"),
                fieldWithPath("inProgress").description("Matches the releases with the IN_PROGRESS status"),
                fieldWithPath("paused").description("Matches the releases with the PAUSED status"),
                fieldWithPath("failing").description("Matches the releases with the FAILING status"),
                fieldWithPath("failed").description("Matches the releases with the FAILED status"),
                fieldWithPath("inactive").description("Matches the releases with the COMPLETED or ABORTED status"),
                fieldWithPath("completed").description("Matches the releases with the COMPLETED status"),
                fieldWithPath("aborted").description("Matches the releases with the ABORTED status"),
                fieldWithPath("onlyMine").description("Matches the releases with me as the owner"),
                fieldWithPath("onlyFlagged").description("Matches the releases which need attention or are at risk"),
                fieldWithPath("onlyArchived").description("Matches the releases which have been archived"),
                fieldWithPath("parentId").type("String").description("Matches the releases stored under this folder"),
                fieldWithPath("templateId").type("String").description("Matches the identifier of the template that the release was created from; for example, Applications/Folder1/Release1"),
                fieldWithPath("kind").type("String").description("Matches the releases with the WORKFLOW or RELEASE kind; default value is RELEASE").optional(),
                fieldWithPath("owner").type("String").description("Matches the releases with the owner username").optional(),
                fieldWithPath("orderBy").type("String").description("The order of the returning set: risk, start_date, end_date, title (only available for templates)").optional(),
                fieldWithPath("orderDirection").type("String").description("The order direction of the returning set: ASC, DESC").optional()
        );
    }

    protected String toJson(ConfigurationItem ci) throws Exception {
        final ByteArrayOutputStream byteArrayOutputStream = new ByteArrayOutputStream();
        configurationItemReaderWriter.writeTo(ci, null, null, null, MediaType.APPLICATION_JSON_TYPE, null, byteArrayOutputStream);
        return new String(byteArrayOutputStream.toByteArray(), "UTF-8");
    }

    protected String toJsonWithPlainTextPassword(ConfigurationItem ci) throws Exception {
        final ByteArrayOutputStream byteArrayOutputStream = new ByteArrayOutputStream();
        nonEncryptingPasswordItemReaderWriter.writeTo(ci, null, null, null, MediaType.APPLICATION_JSON_TYPE, null, byteArrayOutputStream);
        return new String(byteArrayOutputStream.toByteArray(), "UTF-8");
    }

    protected String toJson(List<ConfigurationItem> cis) throws Exception {
        final ByteArrayOutputStream byteArrayOutputStream = new ByteArrayOutputStream();
        configurationItemsReaderWriter.writeTo(cis, null, null, null, MediaType.APPLICATION_JSON_TYPE, null, byteArrayOutputStream);
        return new String(byteArrayOutputStream.toByteArray(), "UTF-8");
    }

    protected String toJson(Object ci) throws Exception {
        return jacksonObjectMapper.writeValueAsString(ci);
    }

    String toJsonWithFields(Object ci) throws Exception {
        ObjectMapper mapper = new ObjectMapper();
        mapper.setVisibility(PropertyAccessor.ALL, Visibility.NONE);
        mapper.setVisibility(PropertyAccessor.FIELD, Visibility.ANY);
        return mapper.writeValueAsString(ci);
    }

    private List<FieldDescriptor> ciFields(Type type) {
        List<FieldDescriptor> fieldDescriptors = new ArrayList<>();
        List<FieldDescriptor> typeFieldDescriptors = type.getDescriptor().getPropertyDescriptors().stream()
                .map(pd -> {
                    FieldDescriptor fieldDescriptor = fieldWithPath(pd.getName()).description(pd.getDescription()).type(determineJsonType(pd));
                    if (!pd.isRequired()) {
                        fieldDescriptor = fieldDescriptor.optional();
                    }
                    if (pd.isHidden() || "internal".equals(pd.getCategory())) {
                        fieldDescriptor = fieldDescriptor.ignored().optional();
                    }
                    return fieldDescriptor;
                })
                .collect(Collectors.toList());
        fieldDescriptors.addAll(basicCiFields(type));
        fieldDescriptors.addAll(typeFieldDescriptors);
        return fieldDescriptors;
    }

    private List<FieldDescriptor> releaseSearch(String database) {
        return newArrayList(
                fieldWithPath("page").description(String.format("The page of Release results from the %s database", database))
                        .type(JsonFieldType.NUMBER),
                fieldWithPath("size").description(String.format("The number of results in this page from the %s database", database))
                        .type(JsonFieldType.NUMBER),
                fieldWithPath("releases").description(String.format("The list of Releases found from the %s database", database))
                        .type(JsonFieldType.ARRAY)
        );
    }

    private List<FieldDescriptor> releaseFields() {
        List<FieldDescriptor> releaseFields = ciFields(Type.valueOf(Release.class));
        releaseFields.addAll(ignoredFields());
        return releaseFields;
    }

    ResponseFieldsSnippet releaseFullSearch = responseFields()
            .andWithPrefix("live.", releaseSearch("active"))
            .andWithPrefix("live.releases[].", releaseFields())
            .andWithPrefix("archived.", releaseSearch("archive"))
            .andWithPrefix("archived.releases[].", releaseFields());

    private List<FieldDescriptor> releaseCount(String suffix) {
        return newArrayList(
                fieldWithPath("total").description(String.format("Number of releases matching the filters, %s", suffix))
                        .type(JsonFieldType.NUMBER),
                fieldWithPath("byStatus").description(String.format("Number of releases matching the filters, by ReleaseStatus and %s", suffix))
                        .type(JsonFieldType.OBJECT)
        );
    }

    private FieldDescriptor countByStatus(ReleaseStatus status, String suffix) {
        return fieldWithPath(status.name()).description(String.format("Number of %s releases matching the filters, %s", status.value(), suffix))
                .type(JsonFieldType.NUMBER)
                .optional();
    }

    private List<FieldDescriptor> releaseCountByStatus(String suffix) {
        return Arrays.stream(ReleaseStatus.values())
                .filter(s -> !s.equals(TEMPLATE))
                .map(s -> countByStatus(s, suffix)).collect(Collectors.toList());
    }

    private String globalSuffix = "from both active and archive databases";
    private String currentSuffix = "from the active database";
    private String archivedSuffix = "from the archive database";

    ResponseFieldsSnippet releaseCountResults = relaxedResponseFields()
            .andWithPrefix("all.", releaseCount(globalSuffix))
            .andWithPrefix("all.byStatus.", releaseCountByStatus(globalSuffix))
            .andWithPrefix("live.", releaseCount(currentSuffix))
            .andWithPrefix("live.byStatus.", releaseCountByStatus(currentSuffix))
            .andWithPrefix("archived.", releaseCount(archivedSuffix))
            .andWithPrefix("archived.byStatus.", releaseCountByStatus(archivedSuffix));


    private List<FieldDescriptor> basicCiFields(Type type) {
        return Arrays.asList(
                fieldWithPath("id").description("This field contains ID of object. " +
                        "It is required but not used on update operations. When creating objects just send 'null'."),
                fieldWithPath("type").description(format("This field represents type of '%s'.", type))
        );
    }

    private List<FieldDescriptor> ignoredFields() {
        return Arrays.asList(
                fieldWithPath("$token").description("The CI optimistic locking hash-tag, please remove it from the API!").ignored().optional(),
                fieldWithPath("$createdAt").description("The date when this CI was created").ignored().optional(),
                fieldWithPath("$createdBy").description("The user that created this CI").ignored().optional(),
                fieldWithPath("$lastModifiedAt").description("The date when this CI was last modified").ignored().optional(),
                fieldWithPath("$lastModifiedBy").description("The last user that modified this CI").ignored().optional(),
                fieldWithPath("$scmTraceabilityDataId").description("The ID of the SCM Data associated to this CI").ignored().optional(),
                fieldWithPath("createdDate").description("The date when this CI was created").ignored().optional(),
                fieldWithPath("modifiedDate").description("The date when this CI was last modified").ignored().optional(),
                fieldWithPath("parentTitle").description("The parent release title for artifacts").ignored().optional()
        );
    }

    private JsonFieldType determineJsonType(PropertyDescriptor pd) {
        switch (pd.getKind()) {
            case BOOLEAN:
                return JsonFieldType.BOOLEAN;
            case CI:
                return JsonFieldType.VARIES;
            case INTEGER:
                return JsonFieldType.NUMBER;
            case ENUM:
                return JsonFieldType.STRING;
            case LIST_OF_CI:
                return JsonFieldType.ARRAY;
            case LIST_OF_STRING:
                return JsonFieldType.ARRAY;
            case SET_OF_CI:
                return JsonFieldType.ARRAY;
            case SET_OF_STRING:
                return JsonFieldType.ARRAY;
            case MAP_STRING_STRING:
                return JsonFieldType.OBJECT;
            default:
                return JsonFieldType.STRING;
        }
    }
}
