package com.xebialabs.xlrelease.utils

import com.xebialabs.xlrelease.domain.events.XLReleaseEvent
import com.xebialabs.xlrelease.events.{SynchronizedSubscribe, XLReleaseEventBus}
import com.xebialabs.xlrelease.utils.MatchingAwaiter.MATCHER
import grizzled.slf4j.Logging

import scala.collection.mutable

/**
  * The idea of this awaiter is to only complete when certain events are complete.
  * It does not want to wrap your code, simply create it before you start your test
  * and it will start collecting all events that are pubished. Once your test is
  * complete simply call .waitFor passing matchers and it will wait for those events
  * that did not appear already
  */
class CleanupAwaiter(eventBus: XLReleaseEventBus, timeout: Long)
  extends BaseConditionAwaiter(eventBus, timeout) with Logging {

  @volatile
  private var remainingMatchers: Seq[MATCHER] = Seq()

  @volatile
  private var events = mutable.ListBuffer[XLReleaseEvent]()

  @SynchronizedSubscribe
  def onEvent(event: XLReleaseEvent): Unit = {
    this.synchronized {
      events += event
      if (remainingMatchers.nonEmpty) {
        remainingMatchers = remainingMatchers.filter(matcher => {
          matcher.apply(event) match {
            case _: MatchingAwaiter.MatchIgnore => true
            case _: MatchingAwaiter.MatchFound => false
            case e: MatchingAwaiter.MatchError => throw new IllegalStateException(e)
            case _ => true
          }
        })
        if (remainingMatchers.isEmpty) {
          latch.countDown()
        }
      }
    }
  }

  def waitFor(matchers: MATCHER*): Unit = {
    this.synchronized {
      remainingMatchers = matchers
          .map(matcher => {
            val wrapped: MATCHER = matcher.orElse {
              case _ => MatchingAwaiter.MatchIgnore()
            }
            wrapped
          })
        .filter(matcher => {
          events.count(event => {
            matcher.apply(event) match {
              case _: MatchingAwaiter.MatchIgnore => false
              case _: MatchingAwaiter.MatchFound => true
              case e: MatchingAwaiter.MatchError => throw new IllegalStateException(e)
              case _ => false
            }
          }) == 0
        })
    }
    if (remainingMatchers.nonEmpty) {
      await()
    }
  }

  override protected def getErrorMessage: String = "Seen these events: " + events.map(_.toString).toSeq.mkString(", ")

}
