package com.xebialabs.xlrelease.utils

import com.xebialabs.xlrelease.actors.ActorSystemHolder
import com.xebialabs.xlrelease.actors.StopProtocol.{RequestStop, Stopped}
import com.xebialabs.xlrelease.utils.ActorTermination.{GiveUpSearch, GiveUpTermination}
import grizzled.slf4j.Logging
import org.apache.pekko.actor.{Actor, ActorIdentity, ActorSystem, Cancellable, Identify, Props}
import org.apache.pekko.event.slf4j.SLF4JLogging

import java.util.UUID
import java.util.concurrent.TimeoutException
import scala.concurrent.duration.{DurationInt, FiniteDuration}
import scala.concurrent.{Await, ExecutionContextExecutor, Promise}

object ActorTermination extends Logging {
  def terminateAllRiskActorsAndAwait(actorSystemHolder: ActorSystemHolder, waitDuration: FiniteDuration): Unit = {
    val actorPath = s"/user/releaseRiskSupervisor"
    val system: ActorSystem = actorSystemHolder.unmanagedActorSystem
    val resultPromise = Promise[String]()
    system.actorOf(props(actorPath, resultPromise))
    Await.result(resultPromise.future, waitDuration)
  }

  def props(path: String, terminatedPromise: Promise[String]) = Props(classOf[ActorTermination], path, terminatedPromise)

  case object GiveUpSearch

  case object GiveUpTermination
}

class ActorTermination(path: String, terminatedPromise: Promise[String]) extends Actor with SLF4JLogging {
  private val msgId = UUID.randomUUID().toString
  private var scheduledMessages: List[Cancellable] = _
  private val searchTimeout: FiniteDuration = 1.second
  private val terminationTimeout: FiniteDuration = 5.second

  override def preStart(): Unit = {
    context.system.actorSelection(path) ! Identify(msgId)
    implicit val ec: ExecutionContextExecutor = context.system.dispatcher
    scheduledMessages = List(
      context.system.scheduler.scheduleOnce(searchTimeout, self, GiveUpSearch),
      context.system.scheduler.scheduleOnce(terminationTimeout, self, GiveUpTermination))
  }

  private def cancelScheduledMessages(): Unit = {
    scheduledMessages.foreach(_.cancel())
  }

  override def receive: Receive = {
    case msg@ActorIdentity(`msgId`, Some(a)) =>
      log.info("Found actor: {}", a.path)
      a ! RequestStop(msgId)
      context.become(waitOnStoppedMsg)
    case ActorIdentity(`msgId`, None) =>
      log.info("Unable to find actor for path: {}", path)
      terminatedPromise.success(s"Actor not found for $path")
      cancelScheduledMessages()
      context.stop(self)
    case GiveUpSearch =>
      terminatedPromise.failure(new TimeoutException(s"Unable to find actor for path: ${path} within $searchTimeout"))
      cancelScheduledMessages()
      context.stop(self)
  }

  private def waitOnStoppedMsg: Receive = {
    case msg@Stopped(`msgId`) =>
      log.info("Received stopped msg from {}", sender.path)
      terminatedPromise.success(s"Actor ${sender.path} stopped")
      context.stop(self)
    case GiveUpTermination =>
      terminatedPromise.failure(new TimeoutException(s"Unable to terminate $path within $terminationTimeout"))
    case GiveUpSearch => ()
  }
}