package com.xebialabs.xlrelease

import com.xebialabs.xlrelease.domain.Team
import com.xebialabs.xlrelease.domain.folder.Folder
import com.xebialabs.xlrelease.repository.{Ids, TeamRepository}
import com.xebialabs.xlrelease.support.pekko.spring.ScalaSpringSupport
import org.scalatest.{Failed, Outcome}
import org.slf4j.MDC
import org.springframework.jdbc.core.JdbcTemplate
import org.springframework.test.context.TestContextManager
import org.springframework.test.context.junit4.rules.ScalaTestSpringClassRule

/**
  * This is a base trait to be inherited by ScalaTest classes which want to run an integration test
  * with XL Release. Inheritors of this class can [[org.springframework.beans.factory.annotation.Autowire]]
  * XL Release services into fields and use them to setup a test environment and assert conditions.
  * <p>
  * A typical test would look like this:
  *
  * {{{
  * class MyIntegrationTest extends XLReleaseIntegrationScalaTest {
  *
  *   describe("MyIntegrationTest") {
  *
  *     it("should create release in repository") {
  *       val release = ReleaseBuilder.newRelease().withId(TestIds.RELEASE1).build();
  *       soreRelease(release)
  *       getRelease(TestIds.RELEASE1) should not be null
  *     }
  *   }
  * }
  * }}}
  *
  * </p>
  * <p>
  * <strong>Note:</strong> A test instance of XL Release is setup in a temporary folder and
  * has some services mocked up for speed and easier testing. So you cannot test initializers or
  * upgraders, for example.
  * </p>
  *
  */
trait XLReleaseIntegrationScalaTest extends XLReleaseIntegrationTest with XLReleaseScalaTest with ScalaSpringSupport {
  val testContextManager: TestContextManager = ScalaTestSpringClassRule.getTestContextManager(this.getClass)

  private lazy val teamRepository = springBean[TeamRepository]

  override protected def beforeAll(): Unit = {
    super.beforeAll()
    testContextManager.beforeTestClass()
    testContextManager.prepareTestInstance(this)
  }

  def getReportingJdbcTemplate: JdbcTemplate = reportingJdbcTemplate

  override def withFixture(test: NoArgTest): Outcome = {
    MDC.put("testName", test.name)
    super.withFixture(test) match {
      case f@Failed(t) =>
        logger.error(s"Test ${test.name} failed", t)
        f
      case other => other
    }
  }

  override protected def beforeEach(): Unit = {
    before()
    super.beforeEach()
  }

  override protected def afterEach(): Unit = {
    try {
      super.afterEach()
      tearDown()
      verifyRepositoryClean()
    } finally {
      MDC.remove("testName")
    }
  }

  override protected def afterAll(): Unit = {
    //    testContextManager.afterTestClass()
    super.afterAll()
  }

  def createTeams(containerId: String, containedTeams: Team*): Seq[Team] = {
    val createdTeams = containedTeams.map(t => {
      teamService.generateIdIfNecessary(containerId, t)
      teamRepository.create(containerId, t)
    })
    createdTeams.foreach(markForDeletion)
    createdTeams
  }

  def createOutOfScopeFolder(title: String): Folder = {
    val folder = new Folder
    folder.setTitle(title)
    createFolder(Ids.ROOT_FOLDER_ID, folder, true)
  }
}
