package com.xebialabs.xlrelease

import com.xebialabs.deployit.exception.NotFoundException
import com.xebialabs.deployit.server.api.upgrade.Version
import com.xebialabs.xlplatform.repository.sql.db.MetadataSchema.METADATA
import com.xebialabs.xlplatform.upgrade.RepositoryVersionService
import com.xebialabs.xlrelease.db.ArchivedReleases._
import com.xebialabs.xlrelease.db.LiquibaseSupport
import com.xebialabs.xlrelease.domain.Team
import com.xebialabs.xlrelease.repository.Ids._
import com.xebialabs.xlrelease.repository._
import com.xebialabs.xlrelease.repository.sql.SqlTeamRepository
import com.xebialabs.xlrelease.repository.sql.persistence.Schema._
import com.xebialabs.xlrelease.repository.sql.persistence.data.FolderRow.Root
import com.xebialabs.xlrelease.repository.sql.persistence.{DependencyPersistence, TaskPersistence}
import com.xebialabs.xlrelease.security.sql.db.SecuritySchema._
import com.xebialabs.xlrelease.service.{ArchivingService, FolderService, ReleaseSearchService}
import com.xebialabs.xlrelease.spring.config.SqlConfiguration
import com.xebialabs.xlrelease.spring.configuration.XlrProfiles
import com.xebialabs.xlrelease.support.akka.spring.ScalaSpringSupport
import com.xebialabs.xlrelease.upgrade.Components.XL_RELEASE_COMPONENT
import com.xebialabs.xlrelease.upgrade.liquibase.BeforeUpgrade
import org.slf4j.MDC
import org.springframework.beans.factory.annotation.{Autowired, Qualifier}
import org.springframework.context.annotation.Profile
import org.springframework.context.{ApplicationContext, ApplicationContextAware}
import org.springframework.jdbc.core.JdbcTemplate
import org.springframework.stereotype.Component

import java.sql.ResultSet
import javax.annotation.PreDestroy
import javax.sql.DataSource
import scala.beans.BeanProperty
import scala.jdk.CollectionConverters._
import scala.util.{Failure, Success, Try}


@Component
@Profile(Array(XlrProfiles.INTEGRATION_TEST))
class SqlStorageFacade @Autowired()(folderService: FolderService,
                                    facetRepository: FacetRepository,
                                    teamRepository: SqlTeamRepository,
                                    releaseRepository: ReleaseRepository,
                                    releaseSearchService: ReleaseSearchService,
                                    dependencyPersistence: DependencyPersistence,
                                    configurationRepository: ConfigurationRepository,
                                    calendarEntryRepository: CalendarEntryRepository,
                                    archivingService: ArchivingService,
                                    triggerRepository: TriggerRepository,
                                    taskPersistence: TaskPersistence,
                                    @Qualifier("xlrRepositoryJdbcTemplate") jdbcTemplate: JdbcTemplate,
                                    @Qualifier("reportingJdbcTemplate") reportingJdbcTemplate: JdbcTemplate)
  extends StorageFacade
    with BeforeUpgrade
    with TestAuthentication
    with LiquibaseSupport
    with ScalaSpringSupport
    with ApplicationContextAware {

  override def xlReleaseVersion(): Version = Version.valueOf(XL_RELEASE_COMPONENT, "0.0.0")

  override def upgrade(): Unit = {
    doUpgrade()
  }

  override def repositoryVersionService: RepositoryVersionService = null // not used by BeforeUpgrade

  override def doUpgrade(): Unit = {
    wipeOutDatabase
    wipeOutArchiveDatabase
  }

  override def countRecordsInTable(tableName: String): Long = {
    jdbcTemplate.queryForObject(s"SELECT count(*) FROM $tableName", classOf[Long])
  }


  private def wipeOutArchiveDatabase: Unit = {
    if (isDbCleanupEnabled) {
      val reportingDataSource = springBean[DataSource]("reportingDataSource")
      val reporting = new LiquibaseSupport {
        override def dataSource: DataSource = reportingDataSource
      }

      reporting.doWithLiquibase("liquibase.xlr.drop.xml") { liquibase =>
        try {
          logger().info("Wipe-out reporting database tables")
          liquibase.dropAll()
        } catch {
          case e: Exception => logger().warn("Unable to wipe-out database", e)
        }
      }
    }
  }

  private def wipeOutDatabase = {
    // Make sure we wipe-out all database tables before next integration test suite runs.
    // Make sure we do not run this if we're not running inside a test (active spring profile must be 'integrationTest')
    // Be careful not to use spring profile "test" as it is used by unit tests and mocks some services.
    if (isDbCleanupEnabled && isIntegrationTestProfileActive) {
      doWithLiquibase("liquibase.xlr.drop.xml") { liquibase =>
        try {
          logger().info("Wipe-out repository database tables")
          liquibase.dropAll()
        } catch {
          case e: Exception => logger().warn("Unable to wipe-out database", e)
        }
      }
    }
  }

  private def isDbCleanupEnabled: Boolean = {
    val nodbcleanup = System.getProperty("nodbcleanup", "false")
    val isCleanupEnabled = !java.lang.Boolean.parseBoolean(nodbcleanup)
    logger().info(s"Cleanup enabled: $isCleanupEnabled")
    isCleanupEnabled
  }

  private def isIntegrationTestProfileActive = {
    applicationContext.getEnvironment.getActiveProfiles.contains(XlrProfiles.INTEGRATION_TEST)
  }

  lazy val dataSource: DataSource = springBean(SqlConfiguration.XLR_REPOSITORY_DATA_SOURCE)

  private val NL = System.lineSeparator

  def tableExists(tableName: String)(implicit jdbcTemplate: JdbcTemplate): Boolean = {
    try {
      jdbcTemplate.queryForObject(s"SELECT count(*) FROM $tableName", classOf[Long])
      true
    } catch {
      case _: Exception => false
    }
  }

  private def deleteTableContent(tableName: String)(implicit jdbcTemplate: JdbcTemplate): Unit = {
    if (tableExists(tableName)) {
      logger().info(s"DELETING $tableName")
      jdbcTemplate.execute(s"DELETE FROM $tableName")
    }
  }

  override def deletePermissionSnapshots(): Unit = {
    implicit val sqlJdbcTemplate: JdbcTemplate = jdbcTemplate
    deleteTableContent(PERMISSION_SNAPSHOTS.TABLE)
  }

  override def cleanup(): Unit = {
    {
      implicit val sqlJdbcTemplate: JdbcTemplate = jdbcTemplate
      deleteTableContent(DEPENDENCIES.TABLE)
      deleteTableContent(ROLE_PERMISSIONS.TABLE)
      deleteTableContent(ROLE_PRINCIPALS.TABLE)
      deleteTableContent(ROLE_ROLES.TABLE)
      deleteTableContent(ROLES.TABLE)
      deleteTableContent(RELEASE_CONFIGURATION_REFS.TABLE)
      deleteTableContent(COMMENTS.TABLE)
      deleteTableContent(TASKS.TABLE)
      deleteTableContent("XLR_ACTIVITY_LOGS")
      deleteTableContent(RELEASES_EXT.TABLE)
      deleteTableContent("XLR_RISK_ASSESSMENTS")
      deleteTableContent("XLR_RISKS")
      deleteTableContent(TRIGGER_CONFIGURATION_REFS.TABLE)
      deleteTableContent(TEMPLATE_TRIGGERS.TABLE)
      deleteTableContent(TRIGGERS.TABLE)
      deleteTableContent(RELEASES.TABLE)
      deleteTableContent(ARTIFACTS.TABLE)
      deleteTableContent(TASK_JOBS.TABLE)
      deleteTableContent(TASK_EXECUTIONS.TABLE)
      deleteTableContent(CATEGORIES.TABLE)
      deletePermissionSnapshots()
      if (tableExists(PATHS.TABLE)) {
        jdbcTemplate.execute(s"DELETE FROM ${PATHS.TABLE} WHERE ${PATHS.ANCESTOR_UID} <> ${Root.uid} OR ${PATHS.DESCENDANT_UID} <> ${Root.uid}")
      }
      if (tableExists(FOLDERS.TABLE)) {
        jdbcTemplate.execute(s"UPDATE ${FOLDERS.TABLE} SET ${FOLDERS.SECURITY_UID} = ${Root.uid}")
        jdbcTemplate.execute(s"DELETE FROM ${FOLDERS.TABLE} WHERE ${FOLDERS.CI_UID} <> ${Root.uid}")
      }
      deleteTableContent(METADATA.TABLE)
      if (tableExists(USER_PROFILE.TABLE)) {
        jdbcTemplate.execute(s"DELETE FROM ${USER_PROFILE.TABLE} WHERE ${USER_PROFILE.USERNAME} <> 'admin'")
      }
    }
    {
      implicit val sqlJdbcTemplate: JdbcTemplate = reportingJdbcTemplate
      deleteTableContent(Archive.ITSM_TASK_REPORTING_RECORD.TABLE)
      deleteTableContent(Archive.DEPLOYMENT_TASK_REPORTING_RECORD.TABLE)
      deleteTableContent(Archive.CODE_COMPLIANCE_TASK_REPORTING_RECORD.TABLE)
      deleteTableContent(REPORT_TASKS_TABLE_NAME)
      deleteTableContent(REPORT_PHASES_TABLE_NAME)
      deleteTableContent(REPORT_RELEASES_TABLE_NAME)
    }
  }

  //noinspection ScalaStyle
  override def delete(id: String): Boolean = Try {
    logger().trace(s"Deleting id $id")
    if (isReleaseId(id)) {
      deleteRelease(id)
    } else if (isDependencyId(id)) {
      dependencyPersistence.deleteDependency(id, taskPersistence.taskUidById(Ids.getParentId(id)).get)
    } else if (isPlanItemId(id)) {
      // ignore
    } else if (isFolderId(id)) {
      if (folderService.exists(id)) {
        folderService.delete(id)
      }
    } else if (isCustomConfigurationId(id)) {
      configurationRepository.delete(id)
    } else if (isCalendarId(id)) {
      calendarEntryRepository.delete(id)
    } else if (isConfigurationId(id) && configurationRepository.exists(id)) {
      configurationRepository.delete(id)
    } else if (isTeamId(id)) {
      teamRepository.delete(id)
    } else if (isVariableId(id)) {
      logger().warn(s"Variable '$id' will be deleted once the parent release is deleted.")
    } else if (id.endsWith("PythonScript") || id.endsWith("/valueProvider")) {
      // Do nothing. Those IDs will be deleted once the parent item is deleted.
    } else if (isTriggerId(id)) {
      triggerRepository.delete(id)
    } else if (isFacetId(id)) {
      facetRepository.delete(id)
    } else if (isInRelease(id)) {
      logger().warn(s"Id '$id' will be deleted once the parent release or template is deleted.")
    } else if (isDeliveryId(id)) {
      deleteDelivery(id)
    } else if (isDashboardId(id)) {
      deleteDashboard(id)
    } else if (isGlobalSettingsId(id)) {
      // Do nothing, these don't get deleted
    } else {
      throw new IllegalArgumentException("Delete of CI with id `%s` is not supported".format(id))
    }
    true
  } match {
    case Success(_) =>
      logger().warn(s"Deleted $id")
      true
    case Failure(_: NotFoundException) =>
      logger().error(s"Unable to delete CI with Id: `$id` as it was not found")
      true
    case Failure(e) =>
      logger().error(s"Unable to delete CI with Id: `$id`", e)
      false
  }

  protected def deleteDelivery(id: String): Unit = {
    // Delivery code is in a bad place for dependencies so hardcoding the query
    jdbcTemplate.execute(s"DELETE FROM XLR_DELIVERIES WHERE ID = \'${getName(id)}\'")
  }

  protected def deleteDashboard(id: String): Unit = {
    // Dashboard code is in a bad place for dependencies so hardcoding the query
    jdbcTemplate.execute(s"DELETE FROM XLR_DASHBOARDS WHERE ID = \'${getName(id)}\'")
  }

  private def isGlobalSettingsId(id: String): Boolean = {
    val shortId = getName(id)
    shortId.startsWith("EmailNotificationSettings")
  }

  protected def deleteRelease(id: String): Unit = {
    logger().info("Deleting release {}", id)
    var found = false
    if (releaseRepository.exists(id)) {
      logger().info("Release {} found in repository", id)
      try {
        teamRepository.deleteTeamsFromPlatform(id)
      } catch {
        case e: NotFoundException => logger().warn(s"Could not delete teams of $id", e)
      }
      releaseRepository.delete(id, failIfReferenced = false)
      found = true
    }

    if (archivingService.exists(id)) {
      logger().info("Release {} found in archive", id)
      deleteFromArchive(id)
      found = true
    }

    if (archivingService.existsPreArchived(id)) {
      logger().info("Release {} found as pre-archived", id)
      deleteFromArchive(id)
      found = true
    }

    if (!found) {
      logger().error("Release {} not found neither in repository nor in archive (including pre-archived releases)", id)
    }
  }

  override def delete(team: Team): Boolean = {
    logger().trace(s"Deleting team ${team.getId}")

    if (!Ids.isInRelease(team.getId) || releaseRepository.exists(Ids.releaseIdFrom(team.getId))) {
      teamRepository.delete(team.getId)
      true
    } else {
      false
    }
  }

  override def deleteFromArchive(releaseId: String): Int =
    SqlStorageFacade.deleteFromArchive(releaseId)(reportingJdbcTemplate)

  override def verifyRepositoryClean(): Unit = {
    val testName = if (MDC.get("testName") != null) s"${MDC.get("testName")}: " else ""
    if (countReleases() > 0 || countArchivedReleases() > 0) {
      throw testDidNotCleanAfterItself(testName, getRemainingReleasesDetails(), getRemainingArchivedReleasesDetails())
    }
    val triggerCount = countTriggers()
    if (triggerCount > 0) {
      throw new TestCleanupException(s"${testName}Found $triggerCount remaining triggers in database:$NL${getRemainingTriggersDetails().mkString(NL)}")
    }
    // also verify that there are no leftover folders
    if (countFolders() > 1 )  {
      val folders = getRemainingFolderDetails().collect { case (id, path) => s"$id -> $path" }
      throw new TestCleanupException(s"${testName}There are leftover folders: ${folders}")
    }
    val jobsCount = countJobs()
    if (jobsCount > 0) {
      val jobs = getRemainingDetails(TASK_JOBS.TABLE, TASK_JOBS.ID, TASK_JOBS.TASK_ID).collect(_._2)
      throw new TestCleanupException(s"${testName}There are leftover jobs: ${jobs}")
    }

    // verify applications, environments and stage
    verifyLeftovers(s"${testName}There are leftover applications", "XLR_APPLICATIONS", "CI_UID", "ID", "TITLE")
    verifyLeftovers(s"${testName}There are leftover environments", "XLR_ENVIRONMENTS", "CI_UID", "ID", "TITLE")
    verifyLeftovers(s"${testName}There are leftover jobs", TASK_JOBS.TABLE, TASK_JOBS.ID, TASK_JOBS.ID, TASK_JOBS.TASK_ID)
  }

  private def countJobs(): Int = {
    jdbcTemplate.queryForObject(
      s"""SELECT COUNT(1) FROM ${TASK_JOBS.TABLE}""",
      (rs: ResultSet, _: Int) => rs.getInt(1)
    )
  }

  private def countFolders(): Int = {
      jdbcTemplate.queryForObject(
        s"""SELECT COUNT(${FOLDERS.CI_UID}) FROM ${FOLDERS.TABLE}""",
        (rs: ResultSet, _: Int) => rs.getInt(1)
      )
  }

  private def countApplications(): Int = {
    jdbcTemplate.queryForObject(
      s"""SELECT COUNT(CI_UID) FROM XLR_APPLICATIONS""",
      (rs: ResultSet, _: Int) => rs.getInt(1)
    )
  }

  private def countRows(tableName: String, columnName: String): Int = {
    jdbcTemplate.queryForObject(
      s"SELECT COUNT($columnName) FROM $tableName",
      (rs: ResultSet, _: Int) => rs.getInt(1)
    )
  }

  private def getRemainingDetails(tableName: String, idColumnName: String, titleColumnName: String): Seq[(String, String)] = {
    jdbcTemplate.query(
      s"SELECT $idColumnName, $titleColumnName FROM $tableName",
      (rs: ResultSet, _: Int) => rs.getString(1) -> rs.getString(2)
    ).asScala.toSeq
  }

  private def verifyLeftovers(message: String, tableName: String, countColumnName: String, idColumnName: String, titleColumnName: String): Unit = {
    val count = countRows(tableName, countColumnName)
    if (count > 1) {
      val leftOvers = getRemainingDetails(tableName, idColumnName, titleColumnName)
      throw new TestCleanupException(s"$message: $leftOvers")
    }
  }

  private def testDidNotCleanAfterItself(testName: String, active: Seq[(String, String)], archived: Seq[(String, String)]) = {
    val numActive = active.size
    val numArchived = archived.size
    val numTotal = numActive + numArchived

    def showEntry(tag: String)(entry: (String, String)): String = entry match {
      case (id, title) => s"$id -> '$title' ($tag)"
    }

    def all = (active.map(showEntry("active")) ++ archived.map(showEntry("archive")))
      .mkString("", "\n * ", "\n")

    new TestCleanupException(s"${testName}Found $numTotal remaining releases ($numActive from database and $numArchived from archive):\n$all")
  }

  private def countReleases(): Int = {
    jdbcTemplate.queryForObject(s"""SELECT COUNT(${RELEASES.CI_UID}) FROM ${RELEASES.TABLE}""", (rs: ResultSet, _: Int) => rs.getInt(1))
  }

  private def countTriggers(): Int = {
    jdbcTemplate.queryForObject(s"""SELECT COUNT(${TRIGGERS.CI_UID}) FROM ${TRIGGERS.TABLE}""", (rs: ResultSet, _: Int) => rs.getInt(1))
  }

  private def getRemainingTriggersDetails(): Seq[String] = {
    jdbcTemplate.query(
      s"""SELECT ${TRIGGERS.TRIGGER_TITLE}, ${TRIGGERS.ID} FROM ${TRIGGERS.TABLE}""",
      (rs: ResultSet, _: Int) => s"Trigger: ${rs.getString(1)} with id: ${rs.getString(2)}"
    ).asScala.toSeq
  }

  private def getRemainingFolderDetails(): Seq[(String, String)] = {
    jdbcTemplate.query(
      s"""SELECT ${FOLDERS.FOLDER_ID}, ${FOLDERS.FOLDER_PATH} FROM ${FOLDERS.TABLE}""",
      (rs: ResultSet, _: Int) => rs.getString(1) -> rs.getString(2)
    ).asScala.toSeq
  }


  private def getRemainingReleasesDetails(): Seq[(String, String)] = {
    jdbcTemplate.query(
      s"""SELECT ${RELEASES.RELEASE_ID}, ${RELEASES.RELEASE_TITLE} FROM ${RELEASES.TABLE}""",
      (rs: ResultSet, _: Int) => rs.getString(1) -> rs.getString(2)
    ).asScala.toSeq
  }

  private def getRemainingArchivedReleasesDetails(): Seq[(String, String)] = {
    reportingJdbcTemplate.query(
      s"""SELECT $REPORT_RELEASES_ID_COLUMN, $REPORT_RELEASES_TITLE_COLUMN FROM $REPORT_RELEASES_TABLE_NAME""",
      (rs: ResultSet, _: Int) => rs.getString(1) -> rs.getString(2)
    ).asScala.toSeq
  }

  private def countArchivedReleases(): Int = {
    reportingJdbcTemplate.queryForObject(
      s"""SELECT COUNT($REPORT_RELEASES_ID_COLUMN) FROM $REPORT_RELEASES_TABLE_NAME""",
      (rs: ResultSet, _: Int) => rs.getInt(1)
    )
  }

  @PreDestroy
  def cleanXlDb(): Unit = {
    implicit val sqlJdbcTemplate: JdbcTemplate = jdbcTemplate
    // delete component versions so that initializers are executed again for integration tests
    deleteTableContent("XL_VERSION")
    wipeOutDatabase
    wipeOutArchiveDatabase
  }

  @BeanProperty
  var applicationContext: ApplicationContext = _
}

object SqlStorageFacade {
  // used statically by JsonUpgradeTest, hence it's here instead of being inside the class.
  def deleteFromArchive(releaseId: String)(jdbcTemplate: JdbcTemplate): Int = {
    jdbcTemplate.update(s"DELETE FROM $REPORT_RELEASES_TABLE_NAME WHERE $REPORT_RELEASES_ID_COLUMN = ?", shortenId(releaseId))
  }
}
