package com.xebialabs.xlrelease.utils;

import java.util.Arrays;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.function.Predicate;
import java.util.function.Supplier;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.xebialabs.xlrelease.db.ArchivedReleases;
import com.xebialabs.xlrelease.domain.Comment;
import com.xebialabs.xlrelease.domain.Release;
import com.xebialabs.xlrelease.domain.Task;
import com.xebialabs.xlrelease.domain.status.ReleaseStatus;
import com.xebialabs.xlrelease.domain.status.TaskStatus;
import com.xebialabs.xlrelease.repository.CommentRepository;
import com.xebialabs.xlrelease.repository.Ids;
import com.xebialabs.xlrelease.repository.ReleaseRepository;

import static scala.jdk.javaapi.CollectionConverters.asJava;

public class Eventually {
    private static final int TIMEOUT_MS = 45000;
    private ReleaseRepository releaseRepository;
    private ArchivedReleases archivedReleases;
    private CommentRepository commentRepository;
    private final long timeoutMs;

    private static final Logger log = LoggerFactory.getLogger(Eventually.class);

    public Eventually(ReleaseRepository releaseRepository, ArchivedReleases archivedReleases, CommentRepository commentRepository) {
        this.releaseRepository = releaseRepository;
        this.archivedReleases = archivedReleases;
        this.commentRepository = commentRepository;
        this.timeoutMs = TIMEOUT_MS;
    }

    public <T extends Task> EventuallyTask<T> task(String taskId) {
        return new EventuallyTask<>(taskId);
    }

    public EventuallyComments comments(String taskId) {
        return new EventuallyComments(taskId);
    }

    public EventuallyRelease release(String releaseId) {
        return new EventuallyRelease(releaseId, 500L);
    }

    public EventuallyRelease release(String releaseId, Long pollingInterval) {
        return new EventuallyRelease(releaseId, pollingInterval);
    }

    public EventuallyArchived archived(String releaseId) { return new EventuallyArchived(releaseId); }

    public <R> R eventually(Supplier<R> supplier, Predicate<R> predicate) {
        return eventually(250L, supplier, predicate);
    }

    public <R> R eventually(Long pollingInterval, Supplier<R> supplier, Predicate<R> predicate) {
        try {
            return CompletableFuture.supplyAsync(() -> {
                        long start = System.currentTimeMillis();
                        Exception lastException = null;
                        while (true) {
                            try {
                                R handle = supplier.get();
                                if (predicate.test(handle)) {
                                    return handle;
                                }
                                long duration = System.currentTimeMillis() - start;
                                if (duration > timeoutMs) {
                                    throw new AssertionError("Could not complete within " + timeoutMs + " millis", lastException);
                                }
                                Thread.sleep(pollingInterval);
                            } catch (Exception store) {
                                lastException = store;
                            }
                        }
                    }
            ).get();
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    public class EventuallyTask<T extends Task> {
        private String taskId;

        public EventuallyTask(final String taskId) {
            this.taskId = taskId;
        }

        public T hasStatus(TaskStatus status) {
            return eventually(() -> (T) releaseRepository.findById(Ids.releaseIdFrom(taskId)).getTask(taskId), (task) -> task.getStatus() == status);
        }

        public T hasOneOfStatus(List<TaskStatus> statuses) {
            return eventually(() -> (T) releaseRepository.findById(Ids.releaseIdFrom(taskId)).getTask(taskId), (task) -> statuses.contains(task.getStatus()));
        }
    }

    public class EventuallyComments {
        private String taskId;

        public EventuallyComments(final String taskId) {
            this.taskId = taskId;
        }

        public List<Comment> hasComments(int numberOfComments) {
            return eventually(() -> asJava(commentRepository.findByTask(taskId)), (comments) -> comments.size() == numberOfComments);
        }
    }

    public class EventuallyRelease {
        private Long pollingInterval = 500L;
        private String releaseId;

        public EventuallyRelease(final String releaseId) {
            this.releaseId = releaseId;
        }

        public EventuallyRelease(final String releaseId, Long pollingInterval) {
            this.releaseId = releaseId;
            this.pollingInterval = pollingInterval;
        }

        public Release hasStatus(ReleaseStatus status) {
            return eventually(pollingInterval, () -> releaseRepository.findById(releaseId), (release) -> release.getStatus() == status);
        }

        public Release hasOneOfStatus(List<ReleaseStatus> statuses) {
            return eventually(pollingInterval, () -> releaseRepository.findById(releaseId), (release) -> statuses.contains(release.getStatus()));
        }

        public Release hasExtensionsLoaded() {
            return eventually(pollingInterval, () -> releaseRepository.findById(releaseId), (release) -> !release.getExtensions().isEmpty());
        }

        public Release hasOneOfStatus(ReleaseStatus... statuses) {
            return hasOneOfStatus(Arrays.asList(statuses));
        }
    }

    public class EventuallyArchived {
        private String releaseId;

        public EventuallyArchived(final String releaseId) {
            this.releaseId = releaseId;
        }

        public boolean isPreArchived() {
            return eventually(() -> archivedReleases.existsPreArchived(releaseId), (preArchived) -> preArchived);
        }

        public boolean isArchived() {
            return eventually(() -> archivedReleases.exists(releaseId), (archived) -> archived);
        }
    }
}
