package com.xebialabs.xlrelease.repository.sql.persistence.configuration

import com.xebialabs.xlrelease.db.sql.SqlBuilder.Dialect
import com.xebialabs.xlrelease.db.sql.transaction.IsTransactional
import com.xebialabs.xlrelease.repository.Ids
import com.xebialabs.xlrelease.repository.sql.persistence.Schema._
import com.xebialabs.xlrelease.repository.sql.persistence.Utils.params
import com.xebialabs.xlrelease.repository.sql.persistence.configuration.ConfigurationPersistence.ConfigurationRow
import com.xebialabs.xlrelease.repository.sql.persistence.data.ConfigurationReferenceRow
import com.xebialabs.xlrelease.repository.sql.persistence.{CiUid, FolderPersistence}
import com.xebialabs.xlrelease.utils.FolderId
import org.springframework.jdbc.core.{JdbcTemplate, RowMapper}

import java.sql.ResultSet
import scala.jdk.CollectionConverters._

@IsTransactional
class ReleaseConfigurationReferencePersistence(override val folderPersistence: FolderPersistence,
                                               override val jdbcTemplate: JdbcTemplate,
                                               override implicit val dialect: Dialect)
  extends ConfigurationReferencePersistenceCommon {

  override protected val table: ConfigurationRefs = RELEASE_CONFIGURATION_REFS

  override protected def STMT_GET_REFS: String =
    s"""|SELECT
        |   entity.${RELEASES.RELEASE_TITLE},
        |   entity.${RELEASES.RELEASE_ID},
        |   folders.${FOLDERS.FOLDER_ID}
        | FROM
        |   ${table.TABLE} refs
        |   JOIN ${RELEASES.TABLE} entity ON refs.${table.USED_BY_UID} = entity.${RELEASES.CI_UID}
        |   JOIN ${FOLDERS.TABLE} folders ON folders.${FOLDERS.CI_UID} = entity.${RELEASES.FOLDER_CI_UID}
        | WHERE
        |   refs.${table.CONFIGURATION_UID} = :configurationCiUid""".stripMargin

  override def findAllByUid(uid: CiUid): Seq[ConfigurationRow] = {
    logger.trace(s"Eager fetching configurations for release with uid: $uid")
    sqlQuery(STMT_ALL_CONFIGURATIONS_BY_UID, params("uid" -> uid), configurationBinaryStreamRowMapper(uid)).toSeq
  }

  override def findAllNonInheritedReferences(folderId: String, queryParameters: Seq[String]): Seq[String] = {
    val sql =
      s"""
         |SELECT DISTINCT
         |  cfg.${CONFIGURATIONS.ID}
         |FROM
         |  ${RELEASES.TABLE} rel
         |  INNER JOIN ${FOLDERS.TABLE} folder ON folder.${FOLDERS.CI_UID} = rel.${RELEASES.FOLDER_CI_UID}
         |  INNER JOIN ${table.TABLE} refs ON refs.${table.USED_BY_UID} = rel.${RELEASES.CI_UID}
         |  INNER JOIN ${CONFIGURATIONS.TABLE} cfg ON cfg.${CONFIGURATIONS.CI_UID} = refs.${table.CONFIGURATION_UID}
         |    AND cfg.${CONFIGURATIONS.FOLDER_CI_UID} IS NOT NULL
         |    AND cfg.${CONFIGURATIONS.FOLDER_CI_UID} != folder.${FOLDERS.CI_UID}
         |WHERE rel.${RELEASES.STATUS} IN (:releaseStatuses)
         |  AND (folder.${FOLDERS.FOLDER_ID} = :folderId OR folder.${FOLDERS.FOLDER_PATH} LIKE :folderPath)
         |""".stripMargin
    val folderPartialId = Ids.getName(folderId)
    val folderPath = s"%${Ids.SEPARATOR}$folderPartialId%"
    sqlQuery(
      sql,
      params("releaseStatuses" -> queryParameters.asJava,
        "folderId" -> folderPartialId,
        "folderPath" -> folderPath
      ),
      rs => rs.getString(CONFIGURATIONS.ID)).toSeq
  }

  override protected def configurationReferenceRowMapper: RowMapper[ConfigurationReferenceRow] = (rs: ResultSet, _: Int) => {
    val title = rs.getString(RELEASES.RELEASE_TITLE)
    val id = rs.getString(RELEASES.RELEASE_ID)
    val folderId = FolderId(rs.getString(FOLDERS.FOLDER_ID))

    ConfigurationReferenceRow(
      title = title,
      id = (folderId / id).absolute
    )
  }
}
