package com.xebialabs.xlrelease.repository.sql

import com.xebialabs.deployit.checks.Checks.IncorrectArgumentException
import com.xebialabs.license.LicenseProperty
import com.xebialabs.license.LicenseVersion4.Edition4
import com.xebialabs.license.service.LicenseService
import com.xebialabs.xlrelease.db.sql.transaction.IsTransactional
import com.xebialabs.xlrelease.domain.status.ReleaseStatus
import com.xebialabs.xlrelease.domain.{Release, ReleaseKind}
import com.xebialabs.xlrelease.exception.LicenseLimitReachedException
import com.xebialabs.xlrelease.limits.LimitEnforcer
import com.xebialabs.xlrelease.limits.LimitEnforcer.LimitType
import com.xebialabs.xlrelease.plugins.dashboard.domain.Dashboard
import com.xebialabs.xlrelease.repository._
import com.xebialabs.xlrelease.repository.sql.ReleasePersistenceInterceptor.{MAX_NO_WORKFLOW_TEMPLATES_ALLOWED, UNRESTRICTED_LICENSE_EDITIONS}
import com.xebialabs.xlrelease.repository.sql.persistence.TaskPersistence
import grizzled.slf4j.Logging
import org.springframework.stereotype.Component
import org.springframework.util.StringUtils

import scala.jdk.CollectionConverters._

@Component
@IsTransactional
class ReleasePersistenceInterceptor(releaseRepository: ReleaseRepository,
                                    val taskPersistence: TaskPersistence,
                                    folderRepository: FolderRepository,
                                    licenseService: LicenseService,
                                    val limitEnforcer: LimitEnforcer)
  extends PersistenceInterceptor[Release]
    with ReleasePhaseTaskLimitValidator
    with WorkflowTaskValidator
    with WorkflowInterceptorLogic
    with Logging {

  releaseRepository.registerPersistenceInterceptor(this)

  override def onCreate(ci: Release): Unit = {
    val tenantId = ensureTenantId(ci)
    enforceTenantLimits(ci, tenantId)
    onCreateOrUpdate(ci, tenantId)
  }

  override def onUpdate(ci: Release): Unit = {
    val tenantId = ensureTenantId(ci)
    onCreateOrUpdate(ci, tenantId)
  }

  private def enforceTenantLimits(ci: Release, tenantId: String): Unit = {
    val (templateLimitType, releaseLimitType, releaseKind) = if (ci.isWorkflow) {
      (LimitType.WORKFLOW_TEMPLATES, LimitType.WORKFLOW_EXECUTIONS, ReleaseKind.WORKFLOW)
    } else {
      (LimitType.TEMPLATES, LimitType.RELEASES, ReleaseKind.RELEASE)
    }

    ci.getStatus match {
      case ReleaseStatus.TEMPLATE =>
        limitEnforcer.enforceLimit(tenantId, templateLimitType, 1, () => tenantTemplateCount(tenantId, releaseKind))
      case ReleaseStatus.PLANNED | ReleaseStatus.IN_PROGRESS | ReleaseStatus.PAUSED | ReleaseStatus.FAILED | ReleaseStatus.FAILING =>
        limitEnforcer.enforceLimit(tenantId, releaseLimitType, 1, () => tenantReleaseCount(tenantId, releaseKind))
      case _ => // No limits to enforce for other statuses
    }
  }

  private def tenantTemplateCount(tenantId: String, releaseKind: ReleaseKind) = {
    releaseRepository.tenantTemplateCountByKind(tenantId, releaseKind)
  }

  private def tenantReleaseCount(tenantId: String, releaseKind: ReleaseKind) = {
    releaseRepository.tenantReleaseCountByKind(tenantId, releaseKind)
  }

  private def onCreateOrUpdate(release: Release, tenantId: String): Unit = {
    enforcePhaseAndTaskLimits(release, tenantId)
    validateTotalNoOfWorkflowTemplates(release)
    updateWorkflowProperties(release)
    validateDefaultTargetFolder(release)
    validateWorkflowProperties(release)
  }

  private def validateDefaultTargetFolder(release: Release): Unit = {
    val doesNotAllowTargetFolderOverride = !release.getAllowTargetFolderOverride
    val defaultTargetFolderId = release.getDefaultTargetFolderId
    val defaultTargetFolderIdIsEmpty = !StringUtils.hasText(defaultTargetFolderId)
    if (release.isTemplate && doesNotAllowTargetFolderOverride && defaultTargetFolderIdIsEmpty) {
      val msg = s"Override of the target folder is not allowed, but default target folder is not provided for release with ID [${release.getId}]"
      throw new IncorrectArgumentException(msg)
    }
    if (!defaultTargetFolderIdIsEmpty) {
      val optionalFolder = folderRepository.findById(defaultTargetFolderId, depth = 1)
      if (optionalFolder.isEmpty) {
        throw new IncorrectArgumentException(s"Cannot find folder with id [$defaultTargetFolderId]")
      }
    }
  }

  private def validateWorkflowProperties(release: Release): Unit = {
    validateNotARootWorkflow(release)
    validateNoDashboardForWorkflow(release)
    validateNoRiskProfileForWorkflow(release)
    validateNoLockTaskPresentForWorkflow(release)
    validateNoUnsupportedTaskPresentForWorkflow(release)
    validateNoMultiLevelNestedTasksForWorkflow(release)
  }

  private def validateNotARootWorkflow(release: Release): Unit = {
    val parentFolder = Ids.getParentId(release.getId)
    if (release.isWorkflow && parentFolder == Ids.ROOT_FOLDER_ID) {
      throw new IncorrectArgumentException("Workflows can be created only inside a folder")
    }
  }

  private def validateNoDashboardForWorkflow(release: Release): Unit = {
    if (release.isWorkflow) {
      val hasDashboards = release.getExtensions.asScala.exists(_.isInstanceOf[Dashboard])
      if (hasDashboards) {
        throw new IncorrectArgumentException("Dashboards are not supported for workflows")
      }
    }
  }

  private def validateNoLockTaskPresentForWorkflow(release: Release): Unit = {
    if (release.isWorkflow) {
      validateNoLockTaskPresent(release.getAllTasks.asScala.toSeq)
    }
  }

  private def validateNoRiskProfileForWorkflow(release: Release): Unit = {
    if (release.isWorkflow && null != release.getRiskProfile) {
      throw new IncorrectArgumentException("Risk profile is not supported for workflows")
    }
  }

  private def validateNoUnsupportedTaskPresentForWorkflow(release: Release): Unit = {
    if (release.isWorkflow) {
      validateNoUnsupportedTaskPresent(release.getAllTasks.asScala.toSeq)
    }
  }

  private def validateNoMultiLevelNestedTasksForWorkflow(release: Release): Unit = {
    if (release.isWorkflow) {
      validateNoMultiLevelNestedTasks(release.getAllTasks.asScala.toSeq)
    }
  }

  private def validateTotalNoOfWorkflowTemplates(release: Release): Unit = {
    if (release.isTemplate && release.isWorkflow) {
      val licenseEdition = getLicenseEdition()
      if (!UNRESTRICTED_LICENSE_EDITIONS.contains(licenseEdition)) {
        val totalWorkflowTemplates = releaseRepository.countTemplatesByKind(ReleaseKind.WORKFLOW)
        if (totalWorkflowTemplates >= MAX_NO_WORKFLOW_TEMPLATES_ALLOWED) {
          throw new LicenseLimitReachedException("License limit reached. " +
            "You have exceeded the limit on the number of workflow templates supported with your license.")
        }
      }
    }
  }

  private def getLicenseEdition(): String = {
    val license = licenseService.getLicense()
    if (license.isDummyLicense) {
      Edition4.Unregistered.name()
    } else {
      // Edition is a required value for all license types
      licenseService.getLicense.getStringValue(LicenseProperty.EDITION)
    }
  }
}

object ReleasePersistenceInterceptor {
  // No of workflow templates allowed without pro or premium edition license
  private val MAX_NO_WORKFLOW_TEMPLATES_ALLOWED: Int = 100

  private val UNRESTRICTED_LICENSE_EDITIONS: Seq[String] = Seq(Edition4.Pro.name(), Edition4.Premium.name())
}
