package com.xebialabs.xlrelease.repository.sql

import com.xebialabs.deployit.booter.local.utils.Strings.isEmpty
import com.xebialabs.deployit.exception.NotFoundException
import com.xebialabs.deployit.security.Permissions.{authenticationToPrincipals, getAuthentication}
import com.xebialabs.deployit.security.{PermissionEnforcer, RoleService}
import com.xebialabs.xlrelease.api.internal.EffectiveSecurityDecorator.EFFECTIVE_SECURITY
import com.xebialabs.xlrelease.api.internal.FolderVariablesDecorator.FOLDER_VARIABLES
import com.xebialabs.xlrelease.api.internal.{DecoratorsCache, EffectiveSecurity, InternalMetadataDecoratorService}
import com.xebialabs.xlrelease.db.sql.SqlBuilder.Dialect
import com.xebialabs.xlrelease.db.sql.transaction.{IsReadOnly, IsTransactional}
import com.xebialabs.xlrelease.domain.folder.Folder
import com.xebialabs.xlrelease.exception.RateLimitReachedException
import com.xebialabs.xlrelease.repository.Ids.ROOT_FOLDER_ID
import com.xebialabs.xlrelease.repository._
import com.xebialabs.xlrelease.repository.sql.persistence.data.FolderRow
import com.xebialabs.xlrelease.repository.sql.persistence.{CiUid, FolderPersistence, ReleasePersistence}
import com.xebialabs.xlrelease.security.XLReleasePermissions.{AUDIT_ALL, EDIT_FOLDER, VIEW_FOLDER}
import com.xebialabs.xlrelease.utils.Tree.Node
import grizzled.slf4j.Logging
import io.micrometer.core.annotation.Timed
import org.springframework.jdbc.core._

import scala.annotation.tailrec
import scala.collection.mutable
import scala.jdk.CollectionConverters._
import scala.util.{Failure, Success, Try}

@IsTransactional
class SqlFolderRepository(folderPersistence: FolderPersistence,
                          releasePersistence: ReleasePersistence,
                          teamRepository: TeamRepository,
                          permissionEnforcer: PermissionEnforcer,
                          roleService: RoleService,
                          val decoratorService: InternalMetadataDecoratorService,
                          implicit val jdbcTemplate: JdbcTemplate,
                          implicit val dialect: Dialect)
  extends FolderRepository
    with SqlRepository
    with Logging {

  @Timed
  @IsReadOnly
  override def exists(folderId: String): Boolean = {
    folderPersistence.exists(folderId)
  }

  @Timed
  @IsReadOnly
  override def getTitle(folderId: String): Option[String] = {
    findById(folderId, depth = 0).map(_.getTitle)
  }

  @Timed
  @IsReadOnly
  override def getPath(folderId: String): Seq[String] = {
    folderPersistence.getFolderPathSegments(folderId)
  }

  @Timed
  @IsReadOnly
  def getUid(folderId: String): CiUid = {
    folderPersistence.getUid(folderId)
  }

  @Timed
  override def findById(folderId: String, depth: Int): Option[Folder] =
    findNodeById(folderId, depth)
      .toOption
      .map { tree =>
        val folder = tree.toFolder
        decoratorService.decorate(folder, Seq(EFFECTIVE_SECURITY, FOLDER_VARIABLES).asJava)
        folder
      }

  @Timed
  override def findViewableFoldersById(folderId: String, depth: Int, enforcePermission: Boolean = true): Option[Folder] = {
    val cache = new DecoratorsCache()
    findViewableNodesById(folderId, depth, enforcePermission)
      .toOption
      .map { tree =>
        val folder = tree.toFolder
        decoratorService.decorate(folder, Seq(EFFECTIVE_SECURITY, FOLDER_VARIABLES).asJava, cache)
        folder.setChildren(decorateWithEffectiveSecurity(folder.getChildren.asScala.toSeq, cache).toSet.asJava)
        folder
      }
  }

  @Timed
  override def listViewableFolders(parentId: String, page: Page, decorateWithPermissions: Boolean = true, enforcePermission: Boolean = true): Seq[Folder] = {
    val viewableResults = paginate(listViewableNodes(parentId, page.depth.intValue(), enforcePermission).map(_.toFolder), page)
    if (decorateWithPermissions) {
      val decoratorsCache = new DecoratorsCache()
      decorateWithEffectiveSecurity(viewableResults, decoratorsCache)
    } else {
      // even we don't want permissions we still need view and edit folder to be used in folders page
      // getting all permissions is very costly operation
      val editableResults = findEditableFoldersById(parentId, page.depth.intValue())
      decorateWithViewPermission(viewableResults)
      decorateWithEditPermission(viewableResults, editableResults.map(_.asFolder.getId).toSeq)
      viewableResults
    }
  }

  @Timed
  override def findNumberOfChildrenForAFolder(folderId: String): Int = {
    folderPersistence.findNumberOfChildrenForAFolder(folderId);
  }

  private def decorateWithEditPermission(folders: Seq[Folder], foldersWithEdit: Seq[String]): Unit = {
    folders.foreach(f => {
      if (foldersWithEdit.contains(f.getId)) {
        val effectiveSecurity = f.get$metadata().get("security").asInstanceOf[EffectiveSecurity]
        effectiveSecurity.getPermissions.add(EDIT_FOLDER.getPermissionName)
      }
      decorateWithEditPermission(f.getChildren.asScala.toSeq, foldersWithEdit)
    })
  }

  private def decorateWithViewPermission(folders: Seq[Folder]): Unit = {
    folders.foreach(f => {
      val permissions = new java.util.HashSet[String]
      permissions.add(VIEW_FOLDER.getPermissionName)
      f.get$metadata().put("security", new EffectiveSecurity(permissions, java.util.Set.of()))
      decorateWithViewPermission(f.getChildren.asScala.toSeq)
    })
  }

  @Timed
  def setAsEffectiveSecurityId(folderId: String): Unit = {
    folderPersistence.setAsEffectiveSecuredCi(folderId)
  }

  @Timed
  def inheritEffectiveSecurityId(folderId: String): Unit = {
    folderPersistence.inheritEffectiveSecuredCi(folderId)
  }

  @Timed
  override def create(parentId: String, folder: Folder): Folder = {
    Try {
      interceptCreate(folder)
      insertNode(folder.getTitle, folder.getId, Some(parentId))
    } match {
      case Success(created) => created.toFolder
      case Failure(e: RateLimitReachedException) => throw e
      case Failure(e) =>
        logger.error(s"Cannot create folder ${folder.getId} under $parentId", e)
        throw new FoldersStoreException(s"Cannot create folder ${folder.getId} under $parentId")
    }
  }

  @Timed
  override def delete(folderId: String, deleteRelease: (Int, String) => Unit): Unit = {
    val ciUid = findNodeById(folderId, 0).value.uid
    folderPersistence.deleteReleases(ciUid, deleteRelease)
    findNodeById(folderId).bottomUp.foreach { data =>
      if (data.hasSecurity) {
        teamRepository.deleteTeamsFromPlatform(data.folderId.absolute)
      }
      folderPersistence.deleteByUid(data.uid)
      interceptDelete(data.folderId.absolute)
    }
  }

  @Timed
  override def rename(folderId: String, newName: String): Folder = {
    folderPersistence.rename(folderId, newName)
  }

  @Timed
  override def move(folderId: String, newParentId: String): Folder = {
    val oldSubtreeSecurityUid = findNodeById(folderId, depth = 0).value.securityUid
    val moved = folderPersistence.move(folderId, newParentId)
    val newSubtreeSecurityUid = moved.value.securityUid
    releasePersistence.replaceSecurityUid(moved.value.uid, oldSubtreeSecurityUid, newSubtreeSecurityUid)
    moved.toFolder
  }

  @Timed
  def findByPath(titlePath: String, depth: Int = Int.MaxValue): Folder = {
    if (isEmpty(titlePath)) {
      throw new NotFoundException(s"Empty path: $titlePath")
    }
    if (titlePath == Ids.SEPARATOR) {
      throw new IllegalArgumentException("Path cannot be empty or root")
    }

    val (dbRow, remaining) = findEntryPoint(titlePath)
      .getOrElse(throw new NotFoundException(s"Path not found: [$titlePath]"))

    val treeDepth = Math.max(depth, titlePath.count(_ == Ids.SEPARATOR.head))
    val treeNode = findNodeByUid(dbRow.uid, treeDepth)

    resolveTreePath(treeNode, remaining, titlePath).toFolder
  }

  private def findEntryPoint(path: String): Option[(FolderRow, String)] = {
    val scanner = PathScanner(path)
    scanner.getSearchCandidates.view.flatMap { candidate =>
      findSubFolderDataByTitle(ROOT_FOLDER_ID, candidate).map { row =>
        val remaining = scanner.path.drop(candidate.length).stripPrefix(Ids.SEPARATOR)
        (row, remaining)
      }
    }.headOption
  }

  @tailrec
  private def resolveTreePath(current: Node[FolderRow], remaining: String, fullPath: String): Node[FolderRow] = {
    if (remaining.isEmpty) return current

    val bestMatch = current.children
      .filter(child => remaining.startsWith(child.value.name))
      .maxByOption(_.value.name.length)

    bestMatch match {
      case Some(node) =>
        val nextPath = remaining.drop(node.value.name.length).stripPrefix(Ids.SEPARATOR)
        resolveTreePath(node, nextPath, fullPath)
      case None =>
        throw new NotFoundException(s"Could not find folder [$remaining] in path [$fullPath]")
    }
  }

  @Timed
  override def findSubFolderByTitle(parentId: String, name: String): Option[Folder] = findSubFolderDataByTitle(parentId, name).map(_.asFolder)

  @Timed
  @IsReadOnly
  override def isFolderInherited(folderId: String): Boolean = {
    folderPersistence.isInherited(folderId)
  }

  @Timed
  @IsReadOnly
  override def tenantFolderCount(tenantId: String): Int = {
    folderPersistence.tenantFolderCount(tenantId)
  }

  @Timed
  @IsReadOnly
  override def getFolderTenant(folderId: String): String = {
    folderPersistence.getFolderTenant(folderId)
  }

  private[sql] def findSubFolderDataByTitle(folderId: String, name: String): Option[FolderRow] = {
    folderPersistence.findSubFolderByTitle(name, folderId)
  }

  private[sql] def insertNode(name: String, givenId: String, parentIdOpt: Option[String]): Node[FolderRow] = {
    folderPersistence.create(name, givenId, parentIdOpt)
  }

  private[sql] def findNodeById(folderId: String, depth: Int = Int.MaxValue): Node[FolderRow] = {
    folderPersistence.findById(folderId, depth)
  }

  private[sql] def findNodeByUid(ciUid: CiUid, depth: Int = Int.MaxValue): Node[FolderRow] = {
    val viewableTree = {
      if (permissionEnforcer.isCurrentUserAdmin || permissionEnforcer.hasLoggedInUserPermission(AUDIT_ALL)) {
        folderPersistence.findByUid(ciUid, depth)
      } else {
        folderPersistence.findByUidHavingPermission(ciUid,
          VIEW_FOLDER,
          authenticationToPrincipals(getAuthentication).asScala,
          getUserRoles(),
          depth)
      }
    }
    viewableTree
  }

  private def listViewableNodes(folderId: String, depth: Int = Int.MaxValue, enforcePermission: Boolean = true): List[Node[FolderRow]] = {
    val viewableTree: Node[FolderRow] = findViewableNodesById(folderId, depth, enforcePermission)
    viewableTree.toOption.fold(List.empty[Node[FolderRow]])(_.children)
  }

  private[sql] def findViewableNodesById(folderId: String, depth: Int = Int.MaxValue, enforcePermission: Boolean = true): Node[FolderRow] = {
    val viewableTree = {
      if (!enforcePermission || (permissionEnforcer.isCurrentUserAdmin || permissionEnforcer.hasLoggedInUserPermission(AUDIT_ALL))) {
        folderPersistence.findById(folderId, depth)
      } else {
        folderPersistence.findByIdHavingPermission(folderId,
          VIEW_FOLDER,
          authenticationToPrincipals(getAuthentication).asScala,
          getUserRoles(),
          depth)
      }
    }
    viewableTree
  }

  // we need flat folders here, if we build the tree nodes without parent they will be dropped
  private[sql] def findEditableFoldersById(folderId: String, depth: Int = Int.MaxValue): mutable.Buffer[FolderRow] = {
    val folders = {
      if (permissionEnforcer.isCurrentUserAdmin || permissionEnforcer.hasLoggedInUserPermission(AUDIT_ALL)) {
        folderPersistence.findDescendantsById(folderId, depth)
      } else {
        folderPersistence.findDescendantsByIdHavingPermission(folderId,
          EDIT_FOLDER,
          authenticationToPrincipals(getAuthentication).asScala,
          getUserRoles(),
          depth)
      }
    }
    folders
  }

  private def getUserRoles() = {
    roleService.getRolesFor(getAuthentication).asScala.map(_.getId)
  }
}

private case class PathScanner(raw: String) {
  val path: String = raw.stripPrefix(Ids.SEPARATOR)
  private val separator = Ids.SEPARATOR.head

  /**
   * Generates possible folder names based on separator:
   * 1. At the end of any alphanumeric segment
   * 2. At any slash that is followed by ANOTHER slash (making it part of a title)
   * 3. At the very end of the string
   */
  def getSearchCandidates: List[String] = {
    (for {
      i <- 1 to path.length
      currentChar = path(i - 1)
      nextChar = if (i < path.length) Some(path(i)) else None

      atFolderEnd = currentChar != separator && nextChar.forall(_ == separator)
      atDataSlash = currentChar == separator && nextChar.contains(separator)
      atStringEnd = i == path.length

      if atFolderEnd || atDataSlash || atStringEnd
    } yield path.substring(0, i)).toList
  }
}
