package com.xebialabs.xlrelease.repository.sql

import com.xebialabs.xlrelease.db.sql.transaction.{IsReadOnly, IsTransactional}
import com.xebialabs.xlrelease.domain.status.PhaseStatus
import com.xebialabs.xlrelease.domain.utils.syntax._
import com.xebialabs.xlrelease.domain.{Phase, Release}
import com.xebialabs.xlrelease.repository.Ids.releaseIdFrom
import com.xebialabs.xlrelease.repository.sql.persistence.CiId._
import com.xebialabs.xlrelease.repository.sql.persistence.DependencyPersistence.TaskDependency
import com.xebialabs.xlrelease.repository.sql.persistence.TaskTagsPersistence.TaskTag
import com.xebialabs.xlrelease.repository.sql.persistence.TasksSqlBuilder.normalizeTag
import com.xebialabs.xlrelease.repository.sql.persistence._
import com.xebialabs.xlrelease.repository.sql.persistence.configuration.ConfigurationReferencePersistence
import com.xebialabs.xlrelease.repository.{DependencyTargetLoader, FacetRepositoryDispatcher, Ids, PhaseRepository}
import com.xebialabs.xlrelease.serialization.json.repository.ResolveOptions
import io.micrometer.core.annotation.Timed

import scala.jdk.CollectionConverters._

@IsTransactional
class SqlPhaseRepository(releasePersistence: ReleasePersistence,
                         phasePersistence: PhasePersistence,
                         taskPersistence: TaskPersistence,
                         val configurationPersistence: ConfigurationReferencePersistence,
                         val commentPersistence: CommentPersistence,
                         val dependencyTargetLoader: DependencyTargetLoader,
                         val dependencyPersistence: DependencyPersistence,
                         val facetRepositoryDispatcher: FacetRepositoryDispatcher,
                         val repositoryAdapter: SqlRepositoryAdapter)
  extends PhaseRepository
    with BaseReleaseItemRepository
    with ConfigurationReferencesSupport {


  @Timed
  def findByIdWithoutDecorators(phaseId: String): Phase = {
    val release = getRelease(releaseIdFrom(phaseId), phaseId, ResolveOptions.WITHOUT_DECORATORS)
    notNull(release.getPhase(phaseId.normalized), phaseId)
  }

  @Timed
  override def findById(phaseId: String): Phase = {
    val release = getRelease(releaseIdFrom(phaseId), phaseId)
    notNull(release.getPhase(phaseId.normalized), phaseId)
  }

  @Timed
  override def create(release: Release, phase: Phase): Phase = {
    interceptCreate(phase)

    releasePersistence.update(release)
    phasePersistence.insert(phase, release.getCiUid)

    val tasks = phase.getAllTasks.asScala
    taskPersistence.batchInsert(tasks.toSet, release.getCiUid)

    // it looks like some databases do not return generated keys after executeBatch is called
    val taskUidsByTaskId: Map[String, TaskCiUid] = taskPersistence.releaseTaskUids(release.getCiUid)
    tasks.foreach(task => task.setCiUid(taskUidsByTaskId(Ids.getFolderlessId(task.getId))))
    val taskTags = for {
      t <- tasks
      tag <- Set.from(t.getTags.asScala.map(normalizeTag))
      taskUid <- taskUidsByTaskId.get(t.getId.shortId)
    } yield TaskTag(taskUid, tag)
    taskPersistence.batchInsertTags(taskTags.toList)

    val filteredTasks = tasks.flatMap(_.dependencies).map(_.getGateTask).toSet
    val taskDependencies = for {
      t <- filteredTasks
      dependency <- t.dependencies.filterNot(_.isArchived)
      taskUid <- taskUidsByTaskId.get(t.getId.shortId)
    } yield TaskDependency(taskUid, dependency)
    dependencyPersistence.batchInsert(taskDependencies)

    updateConfigurationRefs(release)
    facetRepositoryDispatcher.liveRepository.createFromTasks(phase.getAllTasks.asScala.toSeq)
    phase
  }

  @Timed
  override def delete(release: Release, phase: Phase): Phase = {
    // this method is called only from PhaseService.delete
    //  we can expect that ciUid will always be present
    checkIsNotReferencedByDependencies(phase.getId)
    val releaseUid = release.getCiUid
    releasePersistence.updateReleaseJson(releaseUid, release)
    phasePersistence.delete(phase)

    val taskCiUidChunks = phase.getAllTasks.asScala.map(_.getCiUid()).grouped(100)
    taskCiUidChunks.foreach { taskCiUids =>
      dependencyPersistence.deleteByTaskUids(taskCiUids.toSeq)
      commentPersistence.deleteAllByTaskUids(taskCiUids.toSeq)
      taskPersistence.deleteByUids(taskCiUids.toSeq)
    }

    updateConfigurationRefs(release)
    phase
  }

  @Timed
  override def update(release: Release, original: Phase, updated: Phase): Phase = {
    releasePersistence.update(release)
    // phase update never changes underlying tasks or dependencies
    // no need to intercept for that reason
    phasePersistence.update(updated)
    updated
  }

  @Timed
  override def move(release: Release, movedPhase: Phase): Phase = {
    releasePersistence.update(release)
    // phase move operation does not change any phase properties
    // no need to update XLR_PHASES table
    movedPhase
  }

  @Timed
  @IsReadOnly
  override def getTitle(phaseId: String): String = {
    // As of 24.1, system do not extract phases data in separate table for old releases.
    // Kept dependencyTargetLoader for backward compatibility
    phasePersistence.getTitle(phaseId)
      .getOrElse(dependencyTargetLoader.getDependencyNodeData(phaseId).phaseTitle)
  }

  @Timed
  @IsReadOnly
  override def getStatus(phaseId: String): PhaseStatus = {
    // As of 24.1, system do not extract phases data in separate table for old releases.
    // Kept dependencyTargetLoader for backward compatibility
    phasePersistence.getStatus(phaseId)
      .getOrElse(dependencyTargetLoader.getDependencyNodeData(phaseId).phaseStatus)
  }

  @Timed
  override def updateProperties(phase: Phase): Boolean = {
    // phase update properties never changes underlying tasks or dependencies
    // no need to intercept for that reason
    phasePersistence.update(phase)
  }

  @Timed
  override def batchUpdateProperties(phases: Set[Phase]): Int = {
    // phase batch update properties never changes underlying tasks or dependencies
    // no need to intercept for that reason
    if (phases.nonEmpty) {
      phasePersistence.batchUpdate(phases)
    } else {
      0
    }
  }

  @Timed
  override def batchInsertProperties(phases: Set[Phase], releaseUid: Int): Unit = {
    if (phases.nonEmpty) {
      phasePersistence.batchInsert(phases, releaseUid)
    }
  }
}
