package com.xebialabs.xlrelease.repository.sql.security

import com.xebialabs.xlrelease.domain.Team
import com.xebialabs.xlrelease.limits.LimitEnforcer
import com.xebialabs.xlrelease.repository.{FolderRepository, Ids, TeamArrayPropertyChange, TenantLimitRepository}
import com.xebialabs.xlrelease.repository.SecurityRepository.TeamId
import com.xebialabs.xlrelease.security.{XlRolePrincipalRepository, XlRoleRoleRepository}
import com.xebialabs.xlrelease.utils.TenantContext
import grizzled.slf4j.Logging
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.context.annotation.Lazy
import org.springframework.stereotype.Service
import org.springframework.util.StringUtils.hasText

/**
 * Enforcer responsible for team member limits.
 * Handles counting current members, calculating increments, and enforcing limits.
 */
@Service
class TeamMemberLimitEnforcer(limitEnforcer: LimitEnforcer,
                              rolePrincipalsRepository: XlRolePrincipalRepository,
                              roleRolesRepository: XlRoleRoleRepository,
                              tenantLimitRepository: TenantLimitRepository) extends Logging {

  @Autowired
  @Lazy
  private val folderRepository: FolderRepository = null

  private case class MemberCounts(principals: Int, roles: Int) {
    def total: Int = principals + roles
  }

  def enforceLimitForTeamMemberCreation(containerId: String, initialMemberCount: Int): Unit = {
    val tenantId = getTenantId(containerId)
    limitEnforcer.enforceLimit(tenantId, LimitEnforcer.LimitType.TEAM_MEMBERS, 0, () => initialMemberCount)
  }

  def enforceLimitForTeamMemberUpdate(teamId: TeamId, principals: TeamArrayPropertyChange, roles: TeamArrayPropertyChange): Unit = {
    val memberIncrement = (principals.add.size + roles.add.size) - (principals.remove.size + roles.remove.size)
    val isAddingOnly = principals.remove.isEmpty && roles.remove.isEmpty

    if (memberIncrement > 0) {
      enforceLimitForTeamMemberAddition(teamId, memberIncrement, isAddingOnly)
    }
  }

  def enforceLimitForTeamMemberAddition(teamId: TeamId, membersToAdd: Int, isAddingOnly: Boolean = true): Unit = {
    val currentCount = getCurrentMemberCounts(teamId).total
    val tenantId = getTenantId(teamId)
    val limit = getTeamMemberLimit(tenantId)
    val attemptedFinalCount = currentCount + membersToAdd
    val isOverLimit = currentCount >= limit && limit > 0

    // If adding members exceeds the limit, show the limit reached error message with the attempted final state.
    // For example, if the current count is 2 and adding 2 more exceeds the limit of 3,
    // the message should be: "Team members limit reached (4 of 3 used). You cannot add more team members."
    // and not "Team members limit reached (2 of 3 used). You cannot add more team members."
    // This clarifies to the user why the operation was rejected.
    if (isAddingOnly && isOverLimit) {
      limitEnforcer.enforceLimit(tenantId, LimitEnforcer.LimitType.TEAM_MEMBERS, membersToAdd, () => currentCount)
    } else {
      limitEnforcer.enforceLimit(tenantId, LimitEnforcer.LimitType.TEAM_MEMBERS, 0, () => attemptedFinalCount)
    }
  }

  def enforceLimitForBulkTeamMemberCreation(containerId: String, teams: Seq[Team]): Unit = {
    teams.foreach { team =>
      val memberCount = team.getMembers.size() + team.getRoles.size()
      if (memberCount > 0) {
        enforceLimitForTeamMemberCreation(containerId, memberCount)
      }
    }
  }

  private def getCurrentMemberCounts(teamId: TeamId): MemberCounts = {
    val principalCount = rolePrincipalsRepository.findByIdRoleId(teamId).size()
    val roleCount = roleRolesRepository.findByIdRoleId(teamId).size()
    MemberCounts(principalCount, roleCount)
  }

  private def getTeamMemberLimit(tenantId: String): Int = {
    val tenantLimit = tenantLimitRepository.getEffectiveLimit(tenantId)
    if (tenantLimit.isEnabled) {
      val limit = tenantLimit.getMaxTeamMembers
      if (limit < 0) Int.MaxValue else limit
    } else {
      Int.MaxValue // If limits are disabled, return unlimited
    }
  }

  private def getTenantId(teamIdOrContainerId: String): String = {
    val tenantId = TenantContext.getTenant()
    if (hasText(tenantId)) {
      tenantId
    } else {
      val containerId = if (Ids.isTeamId(teamIdOrContainerId)) Ids.getParentId(teamIdOrContainerId) else teamIdOrContainerId
      folderRepository.getFolderTenant(containerId)
    }
  }
}
