package com.xebialabs.xlrelease.repository.sql

import com.xebialabs.deployit.util.PasswordEncrypter
import com.xebialabs.xlrelease.domain.Release
import com.xebialabs.xlrelease.domain.variables._
import com.xebialabs.xlrelease.limits.LimitEnforcer
import com.xebialabs.xlrelease.limits.LimitEnforcer.LimitType

import java.nio.charset.StandardCharsets
import java.util.Optional
import scala.jdk.CollectionConverters._

private[sql] trait VariableSizeLimitValidator {
  def limitEnforcer: LimitEnforcer

  def enforceVariableSizeLimits(ci: Release, tenantId: String): Unit = {
    if (!ci.getStatus.isInactive) {
      // Enforce release variable count limit
      limitEnforcer.enforceLimit(tenantId, LimitType.RELEASE_VARIABLES, 0, () => ci.getVariables.size())
      ci.getVariables.asScala.foreach { variable => enforceVariableSizeLimits(variable, tenantId) }
    }
  }

  def enforceVariableSizeLimits(ci: Variable, tenantId: String): Unit = {
    val variableSize = getVariableValueSize(ci)
    limitEnforcer.enforceLimit(tenantId, LimitType.VARIABLE_VALUE_SIZE, 0, () => variableSize, Optional.of(s"Variable '${ci.getKey}' exceeds the size limit"))
  }

  private def getVariableValueSize(variable: Variable): Int = {
    Option(variable.getValue) match {
      case Some(_) => variable match {
        case stringVar: StringVariable => calculateStringSize(stringVar.getValue)
        case listStringVariable: ListStringVariable =>
          listStringVariable.getValue.asScala.map(calculateStringSize).sum
        case setStringVariable: SetStringVariable =>
          setStringVariable.getValue.asScala.map(calculateStringSize).sum
        case mapStringStringVariable: MapStringStringVariable =>
          mapStringStringVariable.getValue.asScala.map { case (k, v) => calculateStringSize(k) + calculateStringSize(v) }.sum
        case passwordStringVariable: PasswordStringVariable =>
          Option(passwordStringVariable.getValue).map(PasswordEncrypter.getInstance().ensureDecrypted).map(calculateStringSize).sum
        case _ => 0
      }

      case None => 0
    }
  }

  private def calculateStringSize(value: String): Int = {
    Option(value).map(_.getBytes(StandardCharsets.UTF_8).length).getOrElse(0)
  }

}
