package com.xebialabs.xlrelease.repository.sql

import com.xebialabs.xlrelease.db.sql.transaction.IsTransactional
import com.xebialabs.xlrelease.domain.BaseConfiguration
import com.xebialabs.xlrelease.domain.variables.Variable
import com.xebialabs.xlrelease.limits.LimitEnforcer
import com.xebialabs.xlrelease.limits.LimitEnforcer.LimitType
import com.xebialabs.xlrelease.repository.{ConfigurationRepository, PersistenceInterceptor}
import com.xebialabs.xlrelease.utils.TenantContext
import grizzled.slf4j.Logging
import org.springframework.stereotype.Component

/**
 * Global Variable Limit Enforcement Interceptor:
 *
 * Variable Count Limits:
 * - Applied to: GlobalVariables container when variables are added
 * - Scope: Per-tenant
 */
@Component
@IsTransactional
class GlobalVariableLimitPersistenceInterceptor(val limitEnforcer: LimitEnforcer,
                                                configurationRepository: ConfigurationRepository)
  extends PersistenceInterceptor[BaseConfiguration]
    with VariableSizeLimitValidator
    with Logging {

  configurationRepository.registerPersistenceInterceptor(this)

  override def onCreate(ci: BaseConfiguration): Unit = {
    ci match {
      case variable: Variable =>
        logger.debug(s"[GlobalVariableLimitPersistenceInterceptor] Enforcing limits for global variable: ${variable.getKey}")
        val tenantId = TenantContext.getTenant()
        limitEnforcer.enforceLimit(tenantId, LimitType.GLOBAL_VARIABLES, 1, () => configurationRepository.tenantGlobalVariableCount(tenantId))
        enforceVariableSizeLimits(variable, tenantId)
      case _ => ()
    }
  }

  override def onUpdate(ci: BaseConfiguration): Unit = {
    ci match {
      case variable: Variable =>
        logger.debug(s"[GlobalVariableLimitPersistenceInterceptor] Enforcing size limits for global variable update: ${variable.getKey}")
        // Enforce variable size limit for updates (no count limit needed for updates)
        enforceVariableSizeLimits(variable, TenantContext.getTenant())
      case _ => ()
    }
  }
}
